<?php
/**
 * Plugin Name: WC Chatbot
 * Plugin URI: https://woocommercechatbot.com
 * Description: AI-powered sales chatbot for WooCommerce stores
 * Version: 1.95
 * Author: WooCommerce Chatbot
 * Author URI: https://woocommercechatbot.com
 * License: GPL v2 or later
 * Text Domain: wc-chatbot
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 5.0
 */

if (!defined('ABSPATH')) {
    exit;
}

// Plugin constants
define('WCCB_VERSION', '1.95');
define('WCCB_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('WCCB_PLUGIN_URL', plugin_dir_url(__FILE__));
define('WCCB_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Include required files
require_once WCCB_PLUGIN_DIR . 'includes/class-wccb-i18n.php';
require_once WCCB_PLUGIN_DIR . 'includes/class-wccb-api.php';
require_once WCCB_PLUGIN_DIR . 'includes/class-wccb-admin.php';
require_once WCCB_PLUGIN_DIR . 'includes/class-wccb-sync.php';
require_once WCCB_PLUGIN_DIR . 'includes/class-wccb-widget.php';

/**
 * Main plugin class
 */
class WC_Chatbot {

    private static $instance = null;

    public $api;
    public $admin;
    public $sync;
    public $widget;

    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function __construct() {
        WCCB_I18n::init();
        $this->init_hooks();
        $this->init_classes();
    }

    private function init_hooks() {
        add_action('init', [$this, 'load_textdomain']);
        register_activation_hook(__FILE__, [$this, 'activate']);
        register_deactivation_hook(__FILE__, [$this, 'deactivate']);
    }

    private function init_classes() {
        $this->api = new WCCB_API();
        $this->admin = new WCCB_Admin();
        $this->sync = new WCCB_Sync();
        $this->widget = new WCCB_Widget();
    }

    public function load_textdomain() {
        load_plugin_textdomain('wc-chatbot', false, dirname(WCCB_PLUGIN_BASENAME) . '/languages');
    }

    public function activate() {
        // Schedule sync cron job
        if (!wp_next_scheduled('wccb_sync_products')) {
            wp_schedule_event(time(), 'twicedaily', 'wccb_sync_products');
        }

        flush_rewrite_rules();
    }

    public function deactivate() {
        wp_clear_scheduled_hook('wccb_sync_products');
        flush_rewrite_rules();
    }

    /**
     * Get plugin option
     */
    public static function get_option($key, $default = '') {
        $options = get_option('wccb_settings', []);
        return isset($options[$key]) ? $options[$key] : $default;
    }

    /**
     * Update plugin option
     */
    public static function update_option($key, $value) {
        $options = get_option('wccb_settings', []);
        $options[$key] = $value;
        update_option('wccb_settings', $options);
    }

    /**
     * Check if WooCommerce is active
     */
    public static function is_woocommerce_active() {
        return class_exists('WooCommerce');
    }
}

/**
 * Initialize the plugin
 */
function wccb() {
    return WC_Chatbot::instance();
}

// Start the plugin
add_action('plugins_loaded', 'wccb');
