<?php
/**
 * Chat Widget Class
 * Handles the frontend chat widget
 */

if (!defined('ABSPATH')) {
    exit;
}

class WCCB_Widget {

    public function __construct() {
        add_action('wp_footer', [$this, 'render_widget']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_action('wp_ajax_wccb_chat', [$this, 'ajax_chat']);
        add_action('wp_ajax_nopriv_wccb_chat', [$this, 'ajax_chat']);
        add_action('wp_ajax_wccb_track', [$this, 'ajax_track']);
        add_action('wp_ajax_nopriv_wccb_track', [$this, 'ajax_track']);
        add_action('wp_ajax_wccb_track_duration', [$this, 'ajax_track_duration']);
        add_action('wp_ajax_nopriv_wccb_track_duration', [$this, 'ajax_track_duration']);
        add_action('wp_ajax_wccb_poll_chat', [$this, 'ajax_poll_chat']);
        add_action('wp_ajax_nopriv_wccb_poll_chat', [$this, 'ajax_poll_chat']);
        add_action('wp_ajax_wccb_heartbeat', [$this, 'ajax_heartbeat']);
        add_action('wp_ajax_nopriv_wccb_heartbeat', [$this, 'ajax_heartbeat']);
    }

    /**
     * Enqueue frontend scripts
     */
    public function enqueue_scripts() {
        if (!$this->should_show_widget()) {
            return;
        }

        wp_enqueue_style(
            'wccb-widget',
            WCCB_PLUGIN_URL . 'assets/css/widget.css',
            [],
            WCCB_VERSION
        );

        wp_enqueue_script(
            'wccb-widget',
            WCCB_PLUGIN_URL . 'assets/js/widget.js',
            [],
            WCCB_VERSION,
            true
        );

        // Get settings from Hub or use defaults
        $settings = $this->get_widget_settings();

        wp_localize_script('wccb-widget', 'wccb_widget', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('wccb_chat_nonce'),
            'session_id' => $this->get_session_id(),
            'settings' => $settings,
            'design' => [
                'icon_style'       => WC_Chatbot::get_option('widget_icon_style', 'chat'),
                'icon_size'        => intval(WC_Chatbot::get_option('widget_icon_size', 60)),
                'position'         => WC_Chatbot::get_option('widget_position', 'bottom-right'),
                'offset_x'         => intval(WC_Chatbot::get_option('widget_offset_x', 20)),
                'offset_y'         => intval(WC_Chatbot::get_option('widget_offset_y', 20)),
                'chat_width'       => intval(WC_Chatbot::get_option('widget_chat_width', 380)),
                'chat_height'      => intval(WC_Chatbot::get_option('widget_chat_height', 500)),
                'chat_offset_x'    => intval(WC_Chatbot::get_option('widget_chat_offset_x', 0)),
                'chat_offset_y'    => intval(WC_Chatbot::get_option('widget_chat_offset_y', 15)),
                'primary_color'    => WC_Chatbot::get_option('widget_primary_color', '#4F46E5'),
                'bot_bubble_color' => WC_Chatbot::get_option('widget_bot_bubble_color', '#f3f4f6'),
                'header_text_color'=> WC_Chatbot::get_option('widget_header_text_color', '#ffffff'),
                // Mobile overrides
                'mobile_icon_size'      => intval(WC_Chatbot::get_option('widget_mobile_icon_size', 50)),
                'mobile_position'       => WC_Chatbot::get_option('widget_mobile_position', 'bottom-right'),
                'mobile_offset_x'       => intval(WC_Chatbot::get_option('widget_mobile_offset_x', 15)),
                'mobile_offset_y'       => intval(WC_Chatbot::get_option('widget_mobile_offset_y', 15)),
                'mobile_chat_width'     => intval(WC_Chatbot::get_option('widget_mobile_chat_width', 0)),
                'mobile_chat_height'    => intval(WC_Chatbot::get_option('widget_mobile_chat_height', 0)),
                'mobile_chat_offset_x'  => intval(WC_Chatbot::get_option('widget_mobile_chat_offset_x', 0)),
                'mobile_chat_offset_y'  => intval(WC_Chatbot::get_option('widget_mobile_chat_offset_y', 10)),
            ],
            'price_format' => $this->get_wc_price_format(),
            'price_suffix' => WC_Chatbot::get_option('price_suffix_enabled', false) ? WC_Chatbot::get_option('price_suffix', '') : '',
            'strings' => [
                'placeholder' => __('Type your message...', 'wc-chatbot'),
                'send' => __('Send', 'wc-chatbot'),
                'error' => __('Something went wrong. Please try again.', 'wc-chatbot'),
                'view_product' => __('View Product', 'wc-chatbot'),
                'online' => __('Online', 'wc-chatbot'),
            ],
        ]);
    }

    /**
     * Check if widget should be shown
     */
    private function should_show_widget() {
        // Check if widget is enabled
        if (!WC_Chatbot::get_option('widget_enabled', true)) {
            return false;
        }

        // Check if we have a license key
        if (empty(WC_Chatbot::get_option('license_key'))) {
            return false;
        }

        // Don't show in admin
        if (is_admin()) {
            return false;
        }

        return true;
    }

    /**
     * Get widget settings
     */
    private function get_widget_settings() {
        // Try to get cached settings from Hub
        $settings = get_transient('wccb_widget_settings');

        if ($settings === false) {
            // Default settings
            $defaults = [
                'bot_name' => __('Sales Assistant', 'wc-chatbot'),
                'welcome_message' => __('Hi! How can I help you today?', 'wc-chatbot'),
                'primary_color' => '#4F46E5',
                'bot_icon_url' => '',
            ];

            // Try to fetch from Hub
            $result = wccb()->api->get_settings();
            if (!is_wp_error($result) && !empty($result)) {
                $settings = wp_parse_args($result, $defaults);
            } else {
                $settings = $defaults;
            }

            set_transient('wccb_widget_settings', $settings, HOUR_IN_SECONDS);
        }

        // Always override bot_name with local setting (not cached, so changes apply immediately)
        $local_bot_name = WC_Chatbot::get_option('widget_bot_name', '');
        if (!empty($local_bot_name)) {
            $settings['bot_name'] = $local_bot_name;
        }

        return $settings;
    }

    /**
     * Get or create session ID
     */
    private function get_session_id() {
        if (!isset($_COOKIE['wccb_session'])) {
            $session_id = wp_generate_uuid4();
            // Cookie will be set via JavaScript
            return $session_id;
        }
        return sanitize_text_field($_COOKIE['wccb_session']);
    }

    /**
     * Render the widget HTML
     */
    public function render_widget() {
        if (!$this->should_show_widget()) {
            return;
        }

        $settings = $this->get_widget_settings();
        ?>
        <div id="wccb-widget" class="wccb-widget" data-primary-color="<?php echo esc_attr($settings['primary_color']); ?>">
            <!-- Chat Button -->
            <button id="wccb-toggle" class="wccb-toggle" aria-label="<?php esc_attr_e('Open chat', 'wc-chatbot'); ?>">
                <svg class="wccb-icon wccb-icon-chat" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"></path>
                </svg>
                <svg class="wccb-icon wccb-icon-headset" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M3 18v-6a9 9 0 0 1 18 0v6"></path>
                    <path d="M21 19a2 2 0 0 1-2 2h-1a2 2 0 0 1-2-2v-3a2 2 0 0 1 2-2h3zM3 19a2 2 0 0 0 2 2h1a2 2 0 0 0 2-2v-3a2 2 0 0 0-2-2H3z"></path>
                </svg>
                <svg class="wccb-icon wccb-icon-shopping" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M6 2L3 6v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2V6l-3-4z"></path>
                    <line x1="3" y1="6" x2="21" y2="6"></line>
                    <path d="M16 10a4 4 0 0 1-8 0"></path>
                </svg>
                <svg class="wccb-icon wccb-icon-robot" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <rect x="3" y="8" width="18" height="12" rx="2"></rect>
                    <circle cx="9" cy="14" r="1.5" fill="currentColor"></circle>
                    <circle cx="15" cy="14" r="1.5" fill="currentColor"></circle>
                    <path d="M12 2v4"></path>
                    <circle cx="12" cy="2" r="1"></circle>
                </svg>
                <svg class="wccb-icon wccb-icon-support" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"></path>
                    <circle cx="12" cy="7" r="4"></circle>
                    <path d="M5.5 10.5L3 12v3h3l2-2"></path>
                    <path d="M18.5 10.5L21 12v3h-3l-2-2"></path>
                </svg>
                <svg class="wccb-icon-close" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <line x1="18" y1="6" x2="6" y2="18"></line>
                    <line x1="6" y1="6" x2="18" y2="18"></line>
                </svg>
            </button>

            <!-- Chat Window -->
            <div id="wccb-chat" class="wccb-chat">
                <div class="wccb-header">
                    <?php if (!empty($settings['bot_icon_url'])): ?>
                        <img src="<?php echo esc_url($settings['bot_icon_url']); ?>" alt="" class="wccb-bot-icon">
                    <?php endif; ?>
                    <span class="wccb-bot-name"><?php echo esc_html($settings['bot_name']); ?></span>
                    <span class="wccb-status"><?php _e('Online', 'wc-chatbot'); ?></span>
                </div>

                <div id="wccb-messages" class="wccb-messages">
                    <div class="wccb-message wccb-bot">
                        <div class="wccb-message-content">
                            <?php echo esc_html($settings['welcome_message']); ?>
                        </div>
                    </div>
                </div>

                <form id="wccb-form" class="wccb-form">
                    <input type="text" id="wccb-input" class="wccb-input"
                           placeholder="<?php esc_attr_e('Type your message...', 'wc-chatbot'); ?>"
                           autocomplete="off" required>
                    <button type="submit" class="wccb-send" aria-label="<?php esc_attr_e('Send', 'wc-chatbot'); ?>">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <line x1="22" y1="2" x2="11" y2="13"></line>
                            <polygon points="22 2 15 22 11 13 2 9 22 2"></polygon>
                        </svg>
                    </button>
                </form>
            </div>
        </div>
        <?php
    }

    /**
     * Get WooCommerce price formatting settings
     */
    private function get_wc_price_format() {
        if (!class_exists('WooCommerce')) {
            return [
                'currency_symbol' => '$',
                'thousand_sep'    => ',',
                'decimal_sep'     => '.',
                'num_decimals'    => 2,
                'currency_pos'    => 'left',
            ];
        }

        return [
            'currency_symbol' => get_woocommerce_currency_symbol(),
            'thousand_sep'    => wc_get_price_thousand_separator(),
            'decimal_sep'     => wc_get_price_decimal_separator(),
            'num_decimals'    => wc_get_price_decimals(),
            'currency_pos'    => get_option('woocommerce_currency_pos', 'left'),
        ];
    }

    /**
     * Extract products mentioned in the AI reply from local WooCommerce data.
     * Uses three strategies: full-text title matching, regex line parsing, and user query search.
     */
    private function extract_products_from_reply($reply, $user_message = '') {
        if (!class_exists('WooCommerce')) {
            error_log('WCCB extract_products: WooCommerce not loaded');
            return [];
        }

        global $wpdb;

        // Get ALL published product titles from the database (fast single query)
        $all_products = $wpdb->get_results(
            "SELECT ID, post_title FROM {$wpdb->posts}
             WHERE post_type = 'product'
             AND post_status = 'publish'"
        );

        if (empty($all_products)) {
            error_log('WCCB extract_products: no products found in DB');
            return [];
        }

        error_log('WCCB extract_products: ' . count($all_products) . ' products in DB, reply length: ' . strlen($reply));

        $suggested = [];
        $found_ids = [];

        // Strategy 1: Direct text matching — check if any product title appears in the reply
        foreach ($all_products as $row) {
            $title = html_entity_decode($row->post_title, ENT_QUOTES | ENT_HTML5, 'UTF-8');

            if (empty($title) || mb_strlen($title) < 4) {
                continue;
            }

            // Direct match (mb_stripos handles UTF-8 and is case-insensitive)
            if (mb_stripos($reply, $title) !== false) {
                $product = wc_get_product($row->ID);
                if ($product) {
                    $suggested[] = $this->format_product($product);
                    $found_ids[] = (int) $row->ID;
                    error_log('WCCB extract_products: MATCHED "' . $title . '" (ID: ' . $row->ID . ')');
                }
                continue;
            }

            // Normalized match: strip accents from both sides
            $title_clean = mb_strtolower(remove_accents($title));
            $reply_clean = mb_strtolower(remove_accents($reply));
            if (mb_strpos($reply_clean, $title_clean) !== false) {
                $product = wc_get_product($row->ID);
                if ($product) {
                    $suggested[] = $this->format_product($product);
                    $found_ids[] = (int) $row->ID;
                    error_log('WCCB extract_products: MATCHED (normalized) "' . $title . '" (ID: ' . $row->ID . ')');
                }
            }
        }

        // Strategy 2: If no matches, parse numbered lines and try WooCommerce search
        if (empty($suggested)) {
            error_log('WCCB extract_products: Strategy 1 found nothing, trying Strategy 2 (regex parse)');

            // ^ ensures we only match numbered items at the start of a line (not prices like $151.754)
            $names = [];
            if (preg_match_all('/^\s*\d+[\.\)]\s*(.+?)(?:\s*[-–—]\s*[Pp]recio|\s*[-–—]\s*[Pp]rice|\s*$)/um', $reply, $matches)) {
                foreach ($matches[1] as $raw) {
                    $name = trim($raw);
                    if (mb_strlen($name) > 5) {
                        $names[] = $name;
                    }
                }
            }

            error_log('WCCB extract_products: regex parsed ' . count($names) . ' names: ' . implode(' | ', $names));

            foreach ($names as $name) {
                $name_clean = html_entity_decode($name, ENT_QUOTES | ENT_HTML5, 'UTF-8');

                $results = wc_get_products([
                    's' => $name_clean,
                    'limit' => 1,
                    'status' => 'publish',
                ]);

                if (!empty($results)) {
                    $product = $results[0];
                    $pid = $product->get_id();
                    if (!in_array($pid, $found_ids)) {
                        $suggested[] = $this->format_product($product);
                        $found_ids[] = $pid;
                        error_log('WCCB extract_products: MATCHED via search "' . $name_clean . '" → ' . $product->get_name() . ' (ID: ' . $pid . ')');
                    }
                } else {
                    error_log('WCCB extract_products: wc_get_products search found NOTHING for "' . $name_clean . '"');
                }
            }
        }

        // Strategy 3: Direct SQL search using cleaned keywords from user message
        if (count($suggested) < 5 && !empty($user_message)) {
            $keywords = $this->extract_search_keywords($user_message);
            error_log('WCCB extract_products: Strategy 3 - keywords from user message: ' . implode(', ', $keywords));

            if (!empty($keywords)) {
                $where = ["post_type = 'product'", "post_status = 'publish'"];
                foreach ($keywords as $kw) {
                    $where[] = $wpdb->prepare("post_title LIKE %s", '%' . $wpdb->esc_like($kw) . '%');
                }

                $sql = "SELECT ID FROM {$wpdb->posts} WHERE " . implode(' AND ', $where) . " LIMIT 10";
                $ids = $wpdb->get_col($sql);
                error_log('WCCB extract_products: Strategy 3 SQL found ' . count($ids) . ' products');

                foreach ($ids as $pid) {
                    if (count($suggested) >= 5) break;
                    $pid = (int) $pid;
                    if (!in_array($pid, $found_ids)) {
                        $product = wc_get_product($pid);
                        if ($product) {
                            $suggested[] = $this->format_product($product);
                            $found_ids[] = $pid;
                            error_log('WCCB extract_products: MATCHED via SQL keyword search (ID: ' . $pid . ') "' . $product->get_name() . '"');
                        }
                    }
                }
            }
        }

        error_log('WCCB extract_products: returning ' . count($suggested) . ' products');
        return array_slice($suggested, 0, 5);
    }

    /**
     * Extract meaningful search keywords from a user message.
     * Removes stop words, punctuation, and applies basic stemming.
     */
    private function extract_search_keywords($text) {
        $text = mb_strtolower(trim($text));
        // Remove punctuation
        $text = preg_replace('/[¿?¡!.,;:()"\'\-]/', ' ', $text);
        // Split into words
        $words = preg_split('/\s+/', $text, -1, PREG_SPLIT_NO_EMPTY);

        // Stop words in Spanish and English
        $stop = [
            'tienes','tiene','tienen','tengo','hay','de','del','la','el','los','las',
            'un','una','unos','unas','para','por','con','sin','que','es','son',
            'esta','estas','este','estos','como','donde','cuando','cual','cuales',
            'me','te','se','le','lo','nos','les','mi','tu','su','al','a','en',
            'y','o','pero','ni','si','no','muy','mas','ya','e','u','ser','estar',
            'hola','busco','necesito','quiero','quisiera','puedo','puede','podria',
            'do','you','have','the','is','are','any','what','which','how',
            'i','we','they','it','can','could','would','should','will','looking','for',
        ];

        $keywords = [];
        foreach ($words as $w) {
            if (mb_strlen($w) < 2 || in_array($w, $stop)) continue;
            // Basic plural stemming: remove trailing 's' (baterias → bateria)
            if (mb_strlen($w) > 3 && mb_substr($w, -1) === 's') {
                $w = mb_substr($w, 0, -1);
            }
            $keywords[] = $w;
        }

        return $keywords;
    }

    /**
     * Format a WooCommerce product for the chat response
     */
    private function format_product($product) {
        $image_id = $product->get_image_id();
        $image_url = $image_id ? wp_get_attachment_image_url($image_id, 'woocommerce_thumbnail') : '';

        return [
            'id' => $product->get_id(),
            'name' => html_entity_decode($product->get_name(), ENT_QUOTES | ENT_HTML5, 'UTF-8'),
            'price' => $product->get_price(),
            'url' => get_permalink($product->get_id()),
            'image' => $image_url ?: '',
        ];
    }

    /**
     * AJAX: Track visitor pageview or chat event
     */
    public function ajax_track() {
        $page_url = isset($_POST['page_url']) ? esc_url_raw($_POST['page_url']) : '';
        $is_chat = isset($_POST['is_chat']) ? intval($_POST['is_chat']) : 0;
        $visit_token = isset($_POST['visit_token']) ? sanitize_text_field($_POST['visit_token']) : '';

        // Get the real visitor IP
        $visitor_ip = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['HTTP_X_REAL_IP'] ?? $_SERVER['REMOTE_ADDR'] ?? '';
        if (strpos($visitor_ip, ',') !== false) {
            $visitor_ip = trim(explode(',', $visitor_ip)[0]);
        }

        wccb()->api->track($page_url, $is_chat, $visitor_ip, $visit_token);

        wp_send_json_success();
    }

    /**
     * AJAX: Update visit duration
     */
    public function ajax_track_duration() {
        $visit_token = isset($_POST['visit_token']) ? sanitize_text_field($_POST['visit_token']) : '';
        $duration = isset($_POST['duration']) ? intval($_POST['duration']) : 0;

        if (!empty($visit_token) && $duration > 0) {
            wccb()->api->track_duration($visit_token, $duration);
        }

        wp_send_json_success();
    }

    /**
     * AJAX: Handle chat messages
     */
    public function ajax_chat() {
        check_ajax_referer('wccb_chat_nonce', 'nonce');

        $message = isset($_POST['message']) ? sanitize_text_field($_POST['message']) : '';
        $session_id = isset($_POST['session_id']) ? sanitize_text_field($_POST['session_id']) : '';

        if (empty($message)) {
            wp_send_json_error(['message' => __('Message is required', 'wc-chatbot')]);
        }

        if (empty($session_id)) {
            $session_id = wp_generate_uuid4();
        }

        // Get the real visitor IP from WordPress (not the Hub server IP)
        $visitor_ip = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['HTTP_X_REAL_IP'] ?? $_SERVER['REMOTE_ADDR'] ?? '';
        if (strpos($visitor_ip, ',') !== false) {
            $visitor_ip = trim(explode(',', $visitor_ip)[0]);
        }

        $result = wccb()->api->chat($message, $session_id, $visitor_ip);

        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }

        // Decode HTML entities in the reply (fixes &#36; → $, etc.)
        $reply = isset($result['reply']) ? html_entity_decode($result['reply'], ENT_QUOTES | ENT_HTML5, 'UTF-8') : '';

        $products = $result['suggested_products'] ?? [];

        // Supplement with local WooCommerce data: always try if < 5 products
        if (count($products) < 5 && !empty($reply)) {
            $local_products = $this->extract_products_from_reply($reply, $message);

            // Merge local products avoiding duplicates by ID
            $existing_ids = array_map(function($p) { return (int)($p['id'] ?? 0); }, $products);
            foreach ($local_products as $lp) {
                if (count($products) >= 5) break;
                $lp_id = (int)($lp['id'] ?? 0);
                if ($lp_id > 0 && !in_array($lp_id, $existing_ids)) {
                    $products[] = $lp;
                    $existing_ids[] = $lp_id;
                }
            }
        }

        $response_data = [
            'reply' => $reply,
            'suggested_products' => $products,
            'session_id' => $session_id,
        ];

        // Pass through human_mode flag from Hub
        if (!empty($result['human_mode'])) {
            $response_data['human_mode'] = true;
        }

        wp_send_json_success($response_data);
    }

    /**
     * AJAX: Heartbeat for visitor presence tracking
     */
    public function ajax_heartbeat() {
        $session_id = isset($_POST['session_id']) ? sanitize_text_field($_POST['session_id']) : '';
        if (empty($session_id)) {
            wp_send_json_success();
            return;
        }

        wccb()->api->heartbeat($session_id);
        wp_send_json_success();
    }

    /**
     * AJAX: Poll for human mode replies
     */
    public function ajax_poll_chat() {
        check_ajax_referer('wccb_chat_nonce', 'nonce');

        $session_id = isset($_POST['session_id']) ? sanitize_text_field($_POST['session_id']) : '';
        if (empty($session_id)) {
            wp_send_json_error(['message' => __('Session ID required', 'wc-chatbot')]);
        }

        $result = wccb()->api->poll_chat($session_id);

        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }

        wp_send_json_success([
            'human_mode' => !empty($result['human_mode']),
            'history' => $result['history'] ?? [],
            'message_count' => $result['message_count'] ?? 0,
        ]);
    }
}
