<?php
/**
 * Product Sync Class
 * Handles syncing WooCommerce products with the Hub
 */

if (!defined('ABSPATH')) {
    exit;
}

class WCCB_Sync {

    public function __construct() {
        // Cron job for automatic sync
        add_action('wccb_sync_products', [$this, 'sync_products']);

        // Sync on product save (with debounce)
        add_action('save_post_product', [$this, 'schedule_sync_on_change'], 10, 3);

        // Sync on product delete
        add_action('before_delete_post', [$this, 'schedule_sync_on_change']);
    }

    /**
     * Schedule sync when product changes
     */
    public function schedule_sync_on_change($post_id, $post = null, $update = null) {
        // Debounce: Schedule sync for 5 minutes from now
        wp_clear_scheduled_hook('wccb_sync_products_delayed');
        wp_schedule_single_event(time() + 300, 'wccb_sync_products_delayed');
    }

    /**
     * Get all WooCommerce products formatted for sync
     */
    public function get_products() {
        if (!WC_Chatbot::is_woocommerce_active()) {
            return [];
        }

        $products = wc_get_products([
            'status' => 'publish',
            'limit' => -1, // Get all products
            'orderby' => 'date',
            'order' => 'DESC',
        ]);

        $formatted = [];

        foreach ($products as $product) {
            $formatted[] = $this->format_product($product);
        }

        return $formatted;
    }

    /**
     * Format a single product for the API
     */
    private function format_product($product) {
        $image_id = $product->get_image_id();
        $image_url = $image_id ? wp_get_attachment_image_url($image_id, 'woocommerce_thumbnail') : '';

        $data = [
            'id' => $product->get_id(),
            'name' => $product->get_name(),
            'price' => $product->get_price(),
            'stock' => $product->get_stock_status(),
            'url' => $product->get_permalink(),
            'image' => $image_url ?: '',
            'categories' => [],
        ];

        // Get categories
        $categories = wp_get_post_terms($product->get_id(), 'product_cat', ['fields' => 'names']);
        if (!is_wp_error($categories)) {
            $data['categories'] = $categories;
        }

        // Handle variable products
        if ($product->is_type('variable')) {
            $data['price_range'] = [
                'min' => $product->get_variation_price('min'),
                'max' => $product->get_variation_price('max'),
            ];
        }

        return $data;
    }

    /**
     * Get total count of published products
     */
    public function get_products_count() {
        if (!WC_Chatbot::is_woocommerce_active()) {
            return 0;
        }

        $ids = wc_get_products([
            'status' => 'publish',
            'limit' => -1,
            'return' => 'ids',
        ]);

        return count($ids);
    }

    /**
     * Get a batch of products
     */
    public function get_products_batch($offset, $limit) {
        if (!WC_Chatbot::is_woocommerce_active()) {
            return [];
        }

        $products = wc_get_products([
            'status' => 'publish',
            'limit' => $limit,
            'offset' => $offset,
            'orderby' => 'date',
            'order' => 'DESC',
        ]);

        $formatted = [];
        foreach ($products as $product) {
            $formatted[] = $this->format_product($product);
        }

        return $formatted;
    }

    /**
     * Sync products with the Hub
     */
    public function sync_products() {
        $products = $this->get_products();

        if (empty($products)) {
            return new WP_Error('no_products', __('No products to sync', 'wc-chatbot'));
        }

        return wccb()->api->sync_products($products);
    }
}
