<?php
/**
 * Internationalization Class
 * Handles plugin translation using PHP arrays + gettext filter
 */

if (!defined('ABSPATH')) {
    exit;
}

class WCCB_I18n {

    private static $translations = null;
    private static $locale = null;

    /**
     * Locale mapping: WP locale → our language file basename
     */
    private static $locale_map = [
        // Spanish
        'es_ES' => 'es_ES', 'es_MX' => 'es_ES', 'es_AR' => 'es_ES', 'es_CL' => 'es_ES',
        'es_CO' => 'es_ES', 'es_PE' => 'es_ES', 'es_VE' => 'es_ES', 'es_GT' => 'es_ES',
        'es_CR' => 'es_ES', 'es_EC' => 'es_ES', 'es_DO' => 'es_ES', 'es_UY' => 'es_ES',
        'es_PR' => 'es_ES',
        // Portuguese
        'pt_BR' => 'pt_BR', 'pt_PT' => 'pt_BR', 'pt_AO' => 'pt_BR',
        // French
        'fr_FR' => 'fr_FR', 'fr_CA' => 'fr_FR', 'fr_BE' => 'fr_FR', 'fr_CH' => 'fr_FR',
        // German
        'de_DE' => 'de_DE', 'de_AT' => 'de_DE', 'de_CH' => 'de_DE', 'de_DE_formal' => 'de_DE',
        'de_CH_informal' => 'de_DE',
        // Italian
        'it_IT' => 'it_IT',
        // Dutch
        'nl_NL' => 'nl_NL', 'nl_NL_formal' => 'nl_NL', 'nl_BE' => 'nl_NL',
        // Chinese
        'zh_CN' => 'zh_CN', 'zh_TW' => 'zh_CN', 'zh_HK' => 'zh_CN',
        // Japanese
        'ja' => 'ja',
        // Korean
        'ko_KR' => 'ko_KR',
        // Arabic
        'ar' => 'ar', 'ar_SA' => 'ar', 'ar_EG' => 'ar', 'ar_MA' => 'ar',
        // Russian
        'ru_RU' => 'ru_RU',
        // Turkish
        'tr_TR' => 'tr_TR',
        // Hindi
        'hi_IN' => 'hi_IN',
        // Polish
        'pl_PL' => 'pl_PL',
    ];

    /**
     * Initialize translations
     */
    public static function init() {
        self::$locale = self::detect_locale();

        // Only hook filter if not English
        if (self::$locale !== 'en_US' && self::$locale !== 'en_GB' && self::$locale !== 'en_AU') {
            add_filter('gettext', [__CLASS__, 'translate'], 10, 3);
        }
    }

    /**
     * Detect the active locale
     * Priority: plugin setting > WordPress locale > en_US
     */
    private static function detect_locale() {
        $override = WC_Chatbot::get_option('plugin_language', 'auto');

        if (!empty($override) && $override !== 'auto') {
            return $override;
        }

        return determine_locale();
    }

    /**
     * Gettext filter callback
     */
    public static function translate($translated, $text, $domain) {
        if ($domain !== 'wc-chatbot') {
            return $translated;
        }

        if (self::$translations === null) {
            self::load_translations();
        }

        return isset(self::$translations[$text]) ? self::$translations[$text] : $translated;
    }

    /**
     * Load translations for the current locale
     */
    private static function load_translations() {
        $file_key = isset(self::$locale_map[self::$locale]) ? self::$locale_map[self::$locale] : null;

        if (empty($file_key)) {
            self::$translations = [];
            return;
        }

        $file = WCCB_PLUGIN_DIR . 'languages/' . $file_key . '.php';

        if (file_exists($file)) {
            $data = include $file;
            self::$translations = is_array($data) ? $data : [];
        } else {
            self::$translations = [];
        }
    }

    /**
     * Get the current active locale
     */
    public static function get_locale() {
        if (self::$locale === null) {
            self::$locale = self::detect_locale();
        }
        return self::$locale;
    }

    /**
     * Get available languages for the settings dropdown
     */
    public static function get_available_languages() {
        return [
            'auto'  => __('Auto (WordPress default)', 'wc-chatbot'),
            'en_US' => 'English',
            'es_ES' => 'Spanish / Español',
            'pt_BR' => 'Portuguese / Português',
            'fr_FR' => 'French / Français',
            'de_DE' => 'German / Deutsch',
            'it_IT' => 'Italian / Italiano',
            'nl_NL' => 'Dutch / Nederlands',
            'zh_CN' => 'Chinese / 中文',
            'ja'    => 'Japanese / 日本語',
            'ko_KR' => 'Korean / 한국어',
            'ar'    => 'Arabic / العربية',
            'ru_RU' => 'Russian / Русский',
            'tr_TR' => 'Turkish / Türkçe',
            'hi_IN' => 'Hindi / हिन्दी',
            'pl_PL' => 'Polish / Polski',
        ];
    }
}
