<?php
/**
 * API Communication Class
 * Handles all communication with the WooCommerce Chatbot Hub
 */

if (!defined('ABSPATH')) {
    exit;
}

class WCCB_API {

    private $hub_url;
    private $access_token;
    private $token_expiry;

    public function __construct() {
        $this->hub_url = WC_Chatbot::get_option('hub_url', 'https://woocommercechatbot.com');
        $this->access_token = get_transient('wccb_access_token');
    }

    /**
     * Perform handshake with the Hub to get access token
     */
    public function handshake() {
        $license_key = WC_Chatbot::get_option('license_key');

        if (empty($license_key)) {
            return new WP_Error('no_license', __('License key not configured', 'wc-chatbot'));
        }

        $response = wp_remote_post($this->hub_url . '/api/v1/handshake', [
            'timeout' => 30,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ],
            'body' => wp_json_encode([
                'license_key' => $license_key,
                'site_url' => home_url(),
                'plugin_version' => WCCB_VERSION,
                'admin_email' => get_option('admin_email'),
                'wp_language' => get_locale(),
                'wp_country' => self::get_wc_country(),
                'price_format' => self::get_wc_price_format(),
                'training_config' => [
                    'language' => WC_Chatbot::get_option('bot_language', ''),
                    'company_name' => WC_Chatbot::get_option('company_name', ''),
                    'brand_name' => WC_Chatbot::get_option('brand_name', ''),
                    'chatbot_objective' => WC_Chatbot::get_option('chatbot_objective', ''),
                    'custom_prompt' => WC_Chatbot::get_option('custom_prompt', ''),
                    'additional_training' => WC_Chatbot::get_option('additional_training', []),
                ],
            ]),
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($status_code !== 200) {
            $error_message = isset($body['error']) ? $body['error'] : __('Connection failed', 'wc-chatbot');
            return new WP_Error('handshake_failed', $error_message);
        }

        // Store access token
        if (isset($body['access_token'])) {
            $expires_in = isset($body['expires_in']) ? $body['expires_in'] : 86400;
            set_transient('wccb_access_token', $body['access_token'], $expires_in - 300); // 5 min buffer
            $this->access_token = $body['access_token'];

            // Store settings from Hub
            if (isset($body['store'])) {
                WC_Chatbot::update_option('store_id', $body['store']['id']);
                WC_Chatbot::update_option('store_status', $body['store']['status']);
                WC_Chatbot::update_option('monthly_limit', $body['store']['monthly_limit']);
                WC_Chatbot::update_option('messages_used', $body['store']['messages_used']);
            }
        }

        return $body;
    }

    /**
     * Test handshake with custom hub_url and license_key (before saving)
     */
    public function test_handshake($hub_url, $license_key) {
        if (empty($license_key)) {
            return new WP_Error('no_license', __('License key not configured', 'wc-chatbot'));
        }

        if (empty($hub_url)) {
            return new WP_Error('no_hub_url', __('Hub URL not configured', 'wc-chatbot'));
        }

        $hub_url = rtrim($hub_url, '/');

        $response = wp_remote_post($hub_url . '/api/v1/handshake', [
            'timeout' => 30,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ],
            'body' => wp_json_encode([
                'license_key' => $license_key,
                'site_url' => home_url(),
                'plugin_version' => WCCB_VERSION,
                'admin_email' => get_option('admin_email'),
                'wp_language' => get_locale(),
                'wp_country' => self::get_wc_country(),
                'price_format' => self::get_wc_price_format(),
                'training_config' => [
                    'language' => WC_Chatbot::get_option('bot_language', ''),
                    'company_name' => WC_Chatbot::get_option('company_name', ''),
                    'brand_name' => WC_Chatbot::get_option('brand_name', ''),
                    'chatbot_objective' => WC_Chatbot::get_option('chatbot_objective', ''),
                    'custom_prompt' => WC_Chatbot::get_option('custom_prompt', ''),
                    'additional_training' => WC_Chatbot::get_option('additional_training', []),
                ],
            ]),
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($status_code !== 200) {
            $error_message = isset($body['error']) ? $body['error'] : __('Connection failed', 'wc-chatbot');
            return new WP_Error('handshake_failed', $error_message);
        }

        return $body;
    }

    /**
     * Ensure we have a valid access token
     */
    private function ensure_token() {
        if (empty($this->access_token)) {
            $result = $this->handshake();
            if (is_wp_error($result)) {
                return $result;
            }
        }
        return true;
    }

    /**
     * Sync products with the Hub
     */
    public function sync_products($products) {
        $token_check = $this->ensure_token();
        if (is_wp_error($token_check)) {
            return $token_check;
        }

        $response = wp_remote_post($this->hub_url . '/api/v1/sync', [
            'timeout' => 120,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'Authorization' => 'Bearer ' . $this->access_token,
            ],
            'body' => wp_json_encode([
                'products' => $products,
            ]),
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        // Token expired, try re-authenticate
        if ($status_code === 401) {
            delete_transient('wccb_access_token');
            $this->access_token = null;
            return $this->sync_products($products); // Retry once
        }

        if ($status_code !== 200) {
            $error_message = isset($body['error']) ? $body['error'] : __('Sync failed', 'wc-chatbot');
            return new WP_Error('sync_failed', $error_message);
        }

        // Update sync timestamp
        WC_Chatbot::update_option('last_sync', current_time('mysql'));
        WC_Chatbot::update_option('products_synced', isset($body['products_count']) ? $body['products_count'] : 0);

        return $body;
    }

    /**
     * Sync products batch with the Hub
     */
    public function sync_products_batch($products, $isFirst, $isLast, $total) {
        $token_check = $this->ensure_token();
        if (is_wp_error($token_check)) {
            return $token_check;
        }

        $response = wp_remote_post($this->hub_url . '/api/v1/sync', [
            'timeout' => 120,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'Authorization' => 'Bearer ' . $this->access_token,
            ],
            'body' => wp_json_encode([
                'products' => $products,
                'is_first' => $isFirst,
                'is_last' => $isLast,
                'total' => $total,
            ]),
        ]);

        if (is_wp_error($response)) {
            error_log('WCCB sync batch error: ' . $response->get_error_message());
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $raw_body = wp_remote_retrieve_body($response);
        $body = json_decode($raw_body, true);

        if ($status_code === 401) {
            delete_transient('wccb_access_token');
            $this->access_token = null;
            return $this->sync_products_batch($products, $isFirst, $isLast, $total);
        }

        if ($status_code !== 200) {
            $error_message = isset($body['error']) ? $body['error'] : __('Sync failed', 'wc-chatbot');
            error_log('WCCB sync batch HTTP ' . $status_code . ': ' . $raw_body);
            return new WP_Error('sync_failed', $error_message);
        }

        return $body;
    }

    /**
     * Send chat message to the Hub
     */
    public function chat($message, $session_id, $visitor_ip = '') {
        $token_check = $this->ensure_token();
        if (is_wp_error($token_check)) {
            return $token_check;
        }

        $response = wp_remote_post($this->hub_url . '/api/v1/chat', [
            'timeout' => 45,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'Authorization' => 'Bearer ' . $this->access_token,
            ],
            'body' => wp_json_encode([
                'message' => $message,
                'session_id' => $session_id,
                'visitor_ip' => $visitor_ip,
            ]),
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        // Token expired
        if ($status_code === 401) {
            delete_transient('wccb_access_token');
            $this->access_token = null;
            return $this->chat($message, $session_id, $visitor_ip); // Retry once
        }

        // Rate limited
        if ($status_code === 429) {
            return new WP_Error('rate_limited', __('Monthly message limit reached', 'wc-chatbot'));
        }

        if ($status_code !== 200) {
            $error_message = isset($body['error']) ? $body['error'] : __('Chat failed', 'wc-chatbot');
            return new WP_Error('chat_failed', $error_message);
        }

        return $body;
    }

    /**
     * Poll chat for human mode replies
     */
    public function poll_chat($session_id) {
        $token_check = $this->ensure_token();
        if (is_wp_error($token_check)) {
            return $token_check;
        }

        $url = $this->hub_url . '/api/v1/poll-chat?session_id=' . urlencode($session_id);

        $response = wp_remote_get($url, [
            'timeout' => 10,
            'headers' => [
                'Accept' => 'application/json',
                'Authorization' => 'Bearer ' . $this->access_token,
            ],
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($status_code === 401) {
            delete_transient('wccb_access_token');
            $this->access_token = null;
            return $this->poll_chat($session_id);
        }

        if ($status_code !== 200) {
            $error_message = isset($body['error']) ? $body['error'] : __('Poll failed', 'wc-chatbot');
            return new WP_Error('poll_failed', $error_message);
        }

        return $body;
    }

    /**
     * Send heartbeat to Hub for visitor presence tracking
     */
    public function heartbeat($session_id) {
        $token_check = $this->ensure_token();
        if (is_wp_error($token_check)) {
            return $token_check;
        }

        $response = wp_remote_post($this->hub_url . '/api/v1/heartbeat', [
            'timeout' => 5,
            'blocking' => false,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'Authorization' => 'Bearer ' . $this->access_token,
            ],
            'body' => wp_json_encode([
                'session_id' => $session_id,
            ]),
        ]);

        return true;
    }

    /**
     * Get widget settings from the Hub
     */
    public function get_settings() {
        $token_check = $this->ensure_token();
        if (is_wp_error($token_check)) {
            return $token_check;
        }

        $response = wp_remote_get($this->hub_url . '/api/v1/settings', [
            'timeout' => 15,
            'headers' => [
                'Accept' => 'application/json',
                'Authorization' => 'Bearer ' . $this->access_token,
            ],
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($status_code !== 200) {
            return new WP_Error('settings_failed', __('Failed to get settings', 'wc-chatbot'));
        }

        return isset($body['settings']) ? $body['settings'] : [];
    }

    /**
     * Generate custom prompt from questionnaire answers
     */
    public function generate_prompt($answers) {
        $token_check = $this->ensure_token();
        if (is_wp_error($token_check)) {
            return $token_check;
        }

        $response = wp_remote_post($this->hub_url . '/api/v1/generate-prompt', [
            'timeout' => 60,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'Authorization' => 'Bearer ' . $this->access_token,
            ],
            'body' => wp_json_encode([
                'answers' => $answers,
            ]),
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($status_code === 401) {
            delete_transient('wccb_access_token');
            $this->access_token = null;
            return $this->generate_prompt($answers);
        }

        if ($status_code !== 200) {
            $error_message = isset($body['error']) ? $body['error'] : __('Prompt generation failed', 'wc-chatbot');
            return new WP_Error('generate_failed', $error_message);
        }

        return $body;
    }

    /**
     * Get chat logs from the Hub
     */
    public function get_chat_logs($page = 1, $per_page = 20) {
        $token_check = $this->ensure_token();
        if (is_wp_error($token_check)) {
            return $token_check;
        }

        $url = $this->hub_url . '/api/v1/chat-logs?page=' . intval($page) . '&per_page=' . intval($per_page);

        $response = wp_remote_get($url, [
            'timeout' => 15,
            'headers' => [
                'Accept' => 'application/json',
                'Authorization' => 'Bearer ' . $this->access_token,
            ],
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($status_code === 401) {
            delete_transient('wccb_access_token');
            $this->access_token = null;
            return $this->get_chat_logs($page, $per_page);
        }

        if ($status_code !== 200) {
            $error_message = isset($body['error']) ? $body['error'] : __('Failed to get chat logs', 'wc-chatbot');
            return new WP_Error('chat_logs_failed', $error_message);
        }

        return $body;
    }

    /**
     * Get a single chat log from the Hub (for live polling)
     */
    public function get_single_chat_log($id) {
        $token_check = $this->ensure_token();
        if (is_wp_error($token_check)) {
            return $token_check;
        }

        $url = $this->hub_url . '/api/v1/chat-log?id=' . intval($id);
        $response = wp_remote_get($url, [
            'timeout' => 10,
            'headers' => [
                'Accept' => 'application/json',
                'Authorization' => 'Bearer ' . $this->access_token,
            ],
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($status_code === 401) {
            delete_transient('wccb_access_token');
            $this->access_token = null;
            return $this->get_single_chat_log($id);
        }

        if ($status_code !== 200) {
            return new WP_Error('fetch_failed', $body['error'] ?? 'Failed');
        }

        return $body;
    }

    /**
     * Delete a chat log from the Hub
     */
    public function delete_chat($chat_log_id) {
        $token_check = $this->ensure_token();
        if (is_wp_error($token_check)) {
            return $token_check;
        }

        $response = wp_remote_post($this->hub_url . '/api/v1/delete-chat', [
            'timeout' => 15,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'Authorization' => 'Bearer ' . $this->access_token,
            ],
            'body' => wp_json_encode([
                'chat_log_id' => intval($chat_log_id),
            ]),
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($status_code === 401) {
            delete_transient('wccb_access_token');
            $this->access_token = null;
            return $this->delete_chat($chat_log_id);
        }

        if ($status_code !== 200) {
            $error_message = isset($body['error']) ? $body['error'] : __('Delete failed', 'wc-chatbot');
            return new WP_Error('delete_failed', $error_message);
        }

        return $body;
    }

    /**
     * Track a visitor pageview or chat event
     */
    public function track($page_url, $is_chat = 0, $visitor_ip = '', $visit_token = '') {
        $token_check = $this->ensure_token();
        if (is_wp_error($token_check)) {
            return $token_check;
        }

        wp_remote_post($this->hub_url . '/api/v1/track', [
            'timeout' => 5,
            'blocking' => false,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'Authorization' => 'Bearer ' . $this->access_token,
            ],
            'body' => wp_json_encode([
                'page_url' => $page_url,
                'is_chat' => intval($is_chat),
                'visitor_ip' => $visitor_ip,
                'visit_token' => $visit_token,
            ]),
        ]);

        return true;
    }

    /**
     * Update visit duration for a tracked pageview
     */
    public function track_duration($visit_token, $duration) {
        $token_check = $this->ensure_token();
        if (is_wp_error($token_check)) {
            return $token_check;
        }

        wp_remote_post($this->hub_url . '/api/v1/track-duration', [
            'timeout' => 5,
            'blocking' => false,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'Authorization' => 'Bearer ' . $this->access_token,
            ],
            'body' => wp_json_encode([
                'visit_token' => $visit_token,
                'duration' => intval($duration),
            ]),
        ]);

        return true;
    }

    /**
     * Get a quick-access URL for passwordless admin panel login
     */
    public function get_quick_access_url() {
        $token_check = $this->ensure_token();
        if (is_wp_error($token_check)) {
            return $token_check;
        }

        $response = wp_remote_post($this->hub_url . '/api/v1/quick-access', [
            'timeout' => 15,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'Authorization' => 'Bearer ' . $this->access_token,
            ],
            'body' => '{}',
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($status_code === 401) {
            delete_transient('wccb_access_token');
            $this->access_token = null;
            return $this->get_quick_access_url();
        }

        if ($status_code !== 200 || empty($body['url'])) {
            return new WP_Error('quick_access_failed', $body['error'] ?? __('Could not generate access link', 'wc-chatbot'));
        }

        return $body;
    }

    /**
     * Get WooCommerce price formatting settings
     */
    private static function get_wc_price_format() {
        if (!class_exists('WooCommerce')) {
            return [];
        }

        $price_suffix = '';
        if (WC_Chatbot::get_option('price_suffix_enabled', false)) {
            $price_suffix = WC_Chatbot::get_option('price_suffix', '');
        }
        // Strip WooCommerce placeholder tags like {price_including_tax}
        $price_suffix = trim(preg_replace('/\{[^}]+\}/', '', $price_suffix));

        return [
            'currency'        => get_woocommerce_currency(),
            'currency_symbol' => get_woocommerce_currency_symbol(),
            'thousand_sep'    => wc_get_price_thousand_separator(),
            'decimal_sep'     => wc_get_price_decimal_separator(),
            'num_decimals'    => wc_get_price_decimals(),
            'currency_pos'    => get_option('woocommerce_currency_pos', 'left'),
            'price_suffix'    => $price_suffix,
        ];
    }

    /**
     * Get WooCommerce base country code
     */
    private static function get_wc_country() {
        if (!class_exists('WooCommerce')) {
            return '';
        }
        $country = get_option('woocommerce_default_country', '');
        // Format is "CL" or "CL:RM" (country:state), extract country only
        if (strpos($country, ':') !== false) {
            $country = explode(':', $country)[0];
        }
        return $country;
    }
}
