<?php
/**
 * Admin Settings Class
 * Handles plugin settings page in WordPress admin
 */

if (!defined('ABSPATH')) {
    exit;
}

class WCCB_Admin {

    public function __construct() {
        add_action('admin_menu', [$this, 'add_menu_page']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_action('wp_ajax_wccb_test_connection', [$this, 'ajax_test_connection']);
        add_action('wp_ajax_wccb_sync_now', [$this, 'ajax_sync_now']);
        add_action('wp_ajax_wccb_sync_batch', [$this, 'ajax_sync_batch']);
        add_action('wp_ajax_wccb_ping_version', [$this, 'ajax_ping_version']);
        add_action('wp_ajax_wccb_generate_prompt', [$this, 'ajax_generate_prompt']);
        add_action('wp_ajax_wccb_fetch_chat_history', [$this, 'ajax_fetch_chat_history']);
        add_action('wp_ajax_wccb_delete_chat', [$this, 'ajax_delete_chat']);
        add_action('wp_ajax_wccb_fetch_single_chat', [$this, 'ajax_fetch_single_chat']);
        add_action('wp_ajax_wccb_quick_access', [$this, 'ajax_quick_access']);
    }

    /**
     * Add menu page
     */
    public function add_menu_page() {
        add_menu_page(
            __('WC Chatbot', 'wc-chatbot'),
            __('WC Chatbot', 'wc-chatbot'),
            'manage_options',
            'wc-chatbot',
            [$this, 'render_settings_page'],
            'dashicons-format-chat',
            3
        );
    }

    /**
     * Register settings
     */
    public function register_settings() {
        register_setting('wccb_settings_group', 'wccb_settings', [
            'sanitize_callback' => [$this, 'sanitize_settings'],
        ]);
    }

    /**
     * Sanitize settings
     */
    public function sanitize_settings($input) {
        $sanitized = [];

        // Preserve hub_url
        $existing = get_option('wccb_settings', []);
        if (isset($existing['hub_url'])) {
            $sanitized['hub_url'] = $existing['hub_url'];
        }

        if (isset($input['license_key'])) {
            $sanitized['license_key'] = sanitize_text_field($input['license_key']);
        }

        $sanitized['widget_enabled'] = !empty($input['widget_enabled']);

        if (isset($input['plugin_language'])) {
            $sanitized['plugin_language'] = sanitize_text_field($input['plugin_language']);
        }
        if (isset($input['store_timezone'])) {
            $sanitized['store_timezone'] = sanitize_text_field($input['store_timezone']);
        }
        $sanitized['price_suffix_enabled'] = !empty($input['price_suffix_enabled']);
        if (isset($input['price_suffix'])) {
            $sanitized['price_suffix'] = sanitize_text_field($input['price_suffix']);
        }
        if (isset($input['bot_language'])) {
            $sanitized['bot_language'] = sanitize_text_field($input['bot_language']);
        }
        if (isset($input['company_name'])) {
            $sanitized['company_name'] = sanitize_text_field($input['company_name']);
        }
        if (isset($input['brand_name'])) {
            $sanitized['brand_name'] = sanitize_text_field($input['brand_name']);
        }
        if (isset($input['chatbot_objective'])) {
            $sanitized['chatbot_objective'] = sanitize_text_field($input['chatbot_objective']);
        }
        if (isset($input['custom_prompt'])) {
            $sanitized['custom_prompt'] = sanitize_textarea_field($input['custom_prompt']);
        }

        // Additional training entries
        if (isset($input['additional_training']) && is_array($input['additional_training'])) {
            $sanitized_training = [];
            foreach ($input['additional_training'] as $entry) {
                $topic = sanitize_text_field($entry['topic'] ?? '');
                $content = sanitize_textarea_field($entry['content'] ?? '');
                if (!empty($topic) || !empty($content)) {
                    $sanitized_training[] = [
                        'topic' => $topic,
                        'content' => $content,
                    ];
                }
            }
            $sanitized['additional_training'] = $sanitized_training;
        } else {
            $existing_settings = get_option('wccb_settings', []);
            if (isset($existing_settings['additional_training'])) {
                $sanitized['additional_training'] = $existing_settings['additional_training'];
            }
        }

        // Widget design settings
        $widget_text_fields = ['widget_icon_style', 'widget_position', 'widget_primary_color', 'widget_bot_bubble_color', 'widget_header_text_color', 'widget_bot_name', 'widget_mobile_position'];
        foreach ($widget_text_fields as $field) {
            if (isset($input[$field])) {
                $sanitized[$field] = sanitize_text_field($input[$field]);
            }
        }
        $widget_int_fields = ['widget_icon_size', 'widget_offset_x', 'widget_offset_y', 'widget_chat_width', 'widget_chat_height', 'widget_chat_offset_x', 'widget_chat_offset_y', 'widget_mobile_icon_size', 'widget_mobile_offset_x', 'widget_mobile_offset_y', 'widget_mobile_chat_width', 'widget_mobile_chat_height', 'widget_mobile_chat_offset_x', 'widget_mobile_chat_offset_y'];
        foreach ($widget_int_fields as $field) {
            if (isset($input[$field])) {
                $sanitized[$field] = intval($input[$field]);
            }
        }

        // Preserve other settings (prefer $input for programmatic updates, fallback to existing)
        $existing = get_option('wccb_settings', []);
        $preserve_keys = ['store_id', 'store_status', 'monthly_limit', 'messages_used', 'last_sync', 'products_synced'];
        foreach ($preserve_keys as $key) {
            if (isset($input[$key])) {
                $sanitized[$key] = $input[$key];
            } elseif (isset($existing[$key])) {
                $sanitized[$key] = $existing[$key];
            }
        }

        return $sanitized;
    }

    /**
     * Enqueue admin scripts
     */
    public function enqueue_scripts($hook) {
        if ($hook !== 'toplevel_page_wc-chatbot') {
            return;
        }

        wp_enqueue_style(
            'select2',
            'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css',
            [],
            '4.1.0'
        );

        wp_enqueue_script(
            'select2',
            'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js',
            ['jquery'],
            '4.1.0',
            true
        );

        wp_enqueue_style(
            'wccb-admin',
            WCCB_PLUGIN_URL . 'assets/css/admin.css',
            ['select2'],
            WCCB_VERSION
        );

        wp_enqueue_script(
            'wccb-admin',
            WCCB_PLUGIN_URL . 'assets/js/admin.js',
            ['jquery', 'select2'],
            WCCB_VERSION,
            true
        );

        wp_localize_script('wccb-admin', 'wccb_admin', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('wccb_admin_nonce'),
            'wc_active' => WC_Chatbot::is_woocommerce_active() ? '1' : '0',
            'timezone' => WC_Chatbot::get_option('store_timezone', 'UTC'),
            'price_suffix' => WC_Chatbot::get_option('price_suffix_enabled', false) ? WC_Chatbot::get_option('price_suffix', '') : '',
            'price_format' => class_exists('WooCommerce') ? [
                'currency_symbol' => get_woocommerce_currency_symbol(),
                'thousand_sep'    => wc_get_price_thousand_separator(),
                'decimal_sep'     => wc_get_price_decimal_separator(),
                'num_decimals'    => wc_get_price_decimals(),
                'currency_pos'    => get_option('woocommerce_currency_pos', 'left'),
            ] : [
                'currency_symbol' => '$',
                'thousand_sep'    => ',',
                'decimal_sep'     => '.',
                'num_decimals'    => 2,
                'currency_pos'    => 'left',
            ],
            'strings' => [
                'testing' => __('Testing connection...', 'wc-chatbot'),
                'syncing' => __('Syncing products...', 'wc-chatbot'),
                'success' => __('Success!', 'wc-chatbot'),
                'error' => __('Error:', 'wc-chatbot'),
                'remove' => __('Remove', 'wc-chatbot'),
                'topic' => __('Topic', 'wc-chatbot'),
                'content' => __('Content', 'wc-chatbot'),
                'search_language' => __('Search for a language...', 'wc-chatbot'),
                'enter_license_first' => __('Enter License Key first', 'wc-chatbot'),
                'preparing_sync' => __('Preparing sync...', 'wc-chatbot'),
                'no_products' => __('No products found to sync', 'wc-chatbot'),
                'generating' => __('Generating...', 'wc-chatbot'),
                'ai_generating' => __('AI is generating your prompt...', 'wc-chatbot'),
                'prompt_generated' => __('Prompt generated! Switching to Main Company Training...', 'wc-chatbot'),
                'prompt_failed' => __('Failed to generate prompt', 'wc-chatbot'),
                'connection_failed' => __('Connection failed', 'wc-chatbot'),
                'generate_prompt_label' => __('Generate Prompt', 'wc-chatbot'),
                'syncing_progress' => __('Syncing %1$s / %2$s...', 'wc-chatbot'),
                'synced_success' => __('Synced %d products successfully!', 'wc-chatbot'),
                'sync_failed_at' => __('Sync failed at', 'wc-chatbot'),
                'no_expiration' => __('No expiration date', 'wc-chatbot'),
                'view' => __('View', 'wc-chatbot'),
                'view_product' => __('View Product', 'wc-chatbot'),
                'items' => __('items', 'wc-chatbot'),
                'connection_error' => __('Connection error', 'wc-chatbot'),
                'history_error' => __('Error loading chat history', 'wc-chatbot'),
                'delete_chat' => __('Delete', 'wc-chatbot'),
                'confirm_delete' => __('Delete this conversation?', 'wc-chatbot'),
                'placeholder_topic' => __('e.g. Return Policy, Warranty Info, Store Hours...', 'wc-chatbot'),
                'placeholder_content' => __('Enter the training information here...', 'wc-chatbot'),
            ],
        ]);
    }

    /**
     * Render settings page
     */
    public function render_settings_page() {
        $plugin_language = WC_Chatbot::get_option('plugin_language', 'auto');
        $store_timezone = WC_Chatbot::get_option('store_timezone', 'UTC');
        $price_suffix_enabled = WC_Chatbot::get_option('price_suffix_enabled', false);
        $price_suffix = WC_Chatbot::get_option('price_suffix', '');
        $license_key = WC_Chatbot::get_option('license_key', '');
        $widget_enabled = WC_Chatbot::get_option('widget_enabled', true);
        $store_status = WC_Chatbot::get_option('store_status', 'unknown');
        $last_sync = WC_Chatbot::get_option('last_sync', '');
        $products_synced = WC_Chatbot::get_option('products_synced', 0);
        $monthly_limit = WC_Chatbot::get_option('monthly_limit', 50);
        $messages_used = WC_Chatbot::get_option('messages_used', 0);
        $bot_language = WC_Chatbot::get_option('bot_language', '');
        $company_name = WC_Chatbot::get_option('company_name', '');
        $brand_name = WC_Chatbot::get_option('brand_name', '');
        $chatbot_objective = WC_Chatbot::get_option('chatbot_objective', '');
        $custom_prompt = WC_Chatbot::get_option('custom_prompt', '');
        $additional_training = WC_Chatbot::get_option('additional_training', []);
        if (!is_array($additional_training)) {
            $additional_training = [];
        }

        // Widget design settings - Desktop
        $w_icon_style = WC_Chatbot::get_option('widget_icon_style', 'chat');
        $w_icon_size = WC_Chatbot::get_option('widget_icon_size', 60);
        $w_position = WC_Chatbot::get_option('widget_position', 'bottom-right');
        $w_offset_x = WC_Chatbot::get_option('widget_offset_x', 20);
        $w_offset_y = WC_Chatbot::get_option('widget_offset_y', 20);
        $w_chat_width = WC_Chatbot::get_option('widget_chat_width', 380);
        $w_chat_height = WC_Chatbot::get_option('widget_chat_height', 500);
        $w_chat_offset_x = WC_Chatbot::get_option('widget_chat_offset_x', 0);
        $w_chat_offset_y = WC_Chatbot::get_option('widget_chat_offset_y', 15);
        // Widget design settings - Mobile
        $wm_icon_size = WC_Chatbot::get_option('widget_mobile_icon_size', 50);
        $wm_position = WC_Chatbot::get_option('widget_mobile_position', 'bottom-right');
        $wm_offset_x = WC_Chatbot::get_option('widget_mobile_offset_x', 15);
        $wm_offset_y = WC_Chatbot::get_option('widget_mobile_offset_y', 15);
        $wm_chat_width = WC_Chatbot::get_option('widget_mobile_chat_width', 0);
        $wm_chat_height = WC_Chatbot::get_option('widget_mobile_chat_height', 0);
        $wm_chat_offset_x = WC_Chatbot::get_option('widget_mobile_chat_offset_x', 0);
        $wm_chat_offset_y = WC_Chatbot::get_option('widget_mobile_chat_offset_y', 10);
        // Colors & other
        $w_primary_color = WC_Chatbot::get_option('widget_primary_color', '#4F46E5');
        $w_bot_bubble_color = WC_Chatbot::get_option('widget_bot_bubble_color', '#f3f4f6');
        $w_header_text_color = WC_Chatbot::get_option('widget_header_text_color', '#ffffff');
        $w_bot_name = WC_Chatbot::get_option('widget_bot_name', '');
        ?>
        <div class="wrap wccb-admin-wrap">
            <div style="display:flex;align-items:center;gap:12px;flex-wrap:wrap;">
                <h1 style="margin:0;"><?php _e('WC Chatbot', 'wc-chatbot'); ?> <span style="font-size:13px;color:#666;font-weight:400;">v<?php echo WCCB_VERSION; ?></span></h1>
                <?php if (!empty($license_key)): ?>
                <button type="button" id="wccb-manage-store" class="button button-primary" style="display:inline-flex;align-items:center;gap:6px;">
                    <span class="dashicons dashicons-external" style="font-size:16px;width:16px;height:16px;margin-top:2px;"></span>
                    <?php _e('Manage Store', 'wc-chatbot'); ?>
                </button>
                <?php endif; ?>
            </div>

            <?php if (!WC_Chatbot::is_woocommerce_active()): ?>
                <div class="notice notice-info" style="border-left-color:#2271b1;">
                    <p><?php _e('WooCommerce not detected. The chatbot will operate as a support and sales assistant based on your training data.', 'wc-chatbot'); ?></p>
                </div>
            <?php endif; ?>

            <!-- Tabs -->
            <nav class="nav-tab-wrapper wccb-tabs">
                <a href="#settings" class="nav-tab nav-tab-active" data-tab="settings"><?php _e('Settings', 'wc-chatbot'); ?></a>
                <a href="#bot-training" class="nav-tab" data-tab="bot-training"><?php _e('Bot Training', 'wc-chatbot'); ?></a>
                <a href="#widget-settings" class="nav-tab" data-tab="widget-settings"><?php _e('Widget Settings', 'wc-chatbot'); ?></a>
                <a href="#chat-history" class="nav-tab" data-tab="chat-history"><?php _e('Chat History', 'wc-chatbot'); ?></a>
            </nav>

            <form method="post" action="options.php">
                <?php settings_fields('wccb_settings_group'); ?>

                <!-- TAB: Settings -->
                <div class="wccb-tab-panel active" id="wccb-tab-settings" data-tab="settings">
                    <div class="wccb-admin-container">
                        <div class="wccb-admin-main">

                            <div class="wccb-card">
                                <h2><?php _e('Connection Settings', 'wc-chatbot'); ?></h2>

                                <table class="form-table">
                                    <tr>
                                        <th scope="row">
                                            <label for="plugin_language"><?php _e('Plugin Language', 'wc-chatbot'); ?></label>
                                        </th>
                                        <td>
                                            <select id="plugin_language" name="wccb_settings[plugin_language]">
                                                <?php foreach (WCCB_I18n::get_available_languages() as $code => $label): ?>
                                                    <option value="<?php echo esc_attr($code); ?>" <?php selected($plugin_language, $code); ?>><?php echo esc_html($label); ?></option>
                                                <?php endforeach; ?>
                                            </select>
                                            <p class="description"><?php _e('Select the language for the plugin interface. "Auto" uses the WordPress language setting.', 'wc-chatbot'); ?></p>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row">
                                            <label for="store_timezone"><?php _e('Timezone', 'wc-chatbot'); ?></label>
                                        </th>
                                        <td>
                                            <select id="store_timezone" name="wccb_settings[store_timezone]">
                                                <?php
                                                $tz_identifiers = DateTimeZone::listIdentifiers();
                                                foreach ($tz_identifiers as $tz) {
                                                    printf(
                                                        '<option value="%s" %s>%s</option>',
                                                        esc_attr($tz),
                                                        selected($store_timezone, $tz, false),
                                                        esc_html($tz)
                                                    );
                                                }
                                                ?>
                                            </select>
                                            <p class="description"><?php _e('Timezone used to display chat history dates.', 'wc-chatbot'); ?></p>
                                        </td>
                                    </tr>
                                    <?php if (WC_Chatbot::is_woocommerce_active()): ?>
                                    <tr>
                                        <th scope="row"><?php _e('Price Suffix', 'wc-chatbot'); ?></th>
                                        <td>
                                            <label>
                                                <input type="checkbox" name="wccb_settings[price_suffix_enabled]" value="1"
                                                    <?php checked($price_suffix_enabled, true); ?>>
                                                <?php _e('Enable price suffix on product cards', 'wc-chatbot'); ?>
                                            </label>
                                            <div style="margin-top:8px;">
                                                <input type="text" id="price_suffix" name="wccb_settings[price_suffix]"
                                                       value="<?php echo esc_attr($price_suffix); ?>" style="width:200px;"
                                                       placeholder="<?php esc_attr_e('e.g. + IVA, + Tax', 'wc-chatbot'); ?>">
                                            </div>
                                            <p class="description"><?php _e('Text appended after prices on product cards.', 'wc-chatbot'); ?></p>
                                        </td>
                                    </tr>
                                    <?php endif; ?>
                                    <tr>
                                        <th scope="row">
                                            <label for="license_key"><?php _e('License Key', 'wc-chatbot'); ?></label>
                                        </th>
                                        <td>
                                            <input type="text" id="license_key" name="wccb_settings[license_key]"
                                                   value="<?php echo esc_attr($license_key); ?>" class="regular-text">
                                            <p class="description"><?php _e('Your unique license key from the Hub', 'wc-chatbot'); ?></p>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row"><?php _e('Verify API', 'wc-chatbot'); ?></th>
                                        <td>
                                            <button type="button" id="wccb-test-connection" class="button button-secondary">
                                                <?php _e('Verify Connection', 'wc-chatbot'); ?>
                                            </button>
                                            <span id="wccb-test-result" class="wccb-result"></span>
                                            <div id="wccb-license-info" style="display:none; margin-top:12px; padding:12px 16px; border-radius:6px; border:1px solid #c3c4c7; background:#f9f9f9;">
                                                <table style="width:100%; border-collapse:collapse;">
                                                    <tr>
                                                        <td style="padding:4px 8px; font-weight:600; width:140px;"><?php _e('License Status:', 'wc-chatbot'); ?></td>
                                                        <td style="padding:4px 8px;"><span id="wccb-info-status"></span></td>
                                                    </tr>
                                                    <tr>
                                                        <td style="padding:4px 8px; font-weight:600;"><?php _e('Expires:', 'wc-chatbot'); ?></td>
                                                        <td style="padding:4px 8px;"><span id="wccb-info-expires"></span></td>
                                                    </tr>
                                                </table>
                                            </div>
                                        </td>
                                    </tr>
                                </table>
                            </div>

                            <?php submit_button(); ?>
                        </div>

                        <div class="wccb-admin-sidebar">
                            <div class="wccb-card wccb-status-card">
                                <h2><?php _e('Status', 'wc-chatbot'); ?></h2>

                                <div class="wccb-status-item">
                                    <span class="wccb-status-label"><?php _e('License Status:', 'wc-chatbot'); ?></span>
                                    <span class="wccb-status-value wccb-status-<?php echo esc_attr($store_status); ?>">
                                        <?php echo esc_html(ucfirst($store_status)); ?>
                                    </span>
                                </div>

                                <div class="wccb-status-item">
                                    <span class="wccb-status-label"><?php _e('Chats:', 'wc-chatbot'); ?></span>
                                    <span class="wccb-status-value" id="wccb-sidebar-chats">
                                        <?php echo esc_html($messages_used); ?> / <?php echo esc_html($monthly_limit); ?>
                                    </span>
                                </div>

                                <?php if (WC_Chatbot::is_woocommerce_active()): ?>
                                <div class="wccb-status-item">
                                    <span class="wccb-status-label"><?php _e('Products Synced:', 'wc-chatbot'); ?></span>
                                    <span class="wccb-status-value"><?php echo esc_html($products_synced); ?></span>
                                </div>

                                <div class="wccb-status-item">
                                    <span class="wccb-status-label"><?php _e('Last Sync:', 'wc-chatbot'); ?></span>
                                    <span class="wccb-status-value">
                                        <?php echo $last_sync ? esc_html($last_sync) : __('Never', 'wc-chatbot'); ?>
                                    </span>
                                </div>

                                <hr>

                                <button type="button" id="wccb-sync-now" class="button button-primary" style="width: 100%;">
                                    <?php _e('Sync Products Now', 'wc-chatbot'); ?>
                                </button>
                                <span id="wccb-sync-result" class="wccb-result"></span>
                                <?php else: ?>
                                <div class="wccb-status-item">
                                    <span class="wccb-status-label"><?php _e('Mode:', 'wc-chatbot'); ?></span>
                                    <span class="wccb-status-value"><?php _e('Training-based', 'wc-chatbot'); ?></span>
                                </div>
                                <hr>
                                <p style="margin:0;color:#666;font-size:12px;line-height:1.5;"><?php _e('The chatbot responds based on your training data. Add training content in the Bot Training tab.', 'wc-chatbot'); ?></p>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- TAB: Bot Training (with vertical sub-tabs) -->
                <div class="wccb-tab-panel" id="wccb-tab-bot-training" data-tab="bot-training" style="display:none;">
                    <div class="wccb-subtab-layout">
                        <!-- Left sidebar: vertical sub-tab navigation -->
                        <div class="wccb-subtab-nav">
                            <a href="#" class="wccb-subtab-link active" data-subtab="main-training"><?php _e('Main Company Training', 'wc-chatbot'); ?></a>
                            <a href="#" class="wccb-subtab-link" data-subtab="generate-prompt"><?php _e('Generate Prompt with AI', 'wc-chatbot'); ?></a>
                            <a href="#" class="wccb-subtab-link" data-subtab="add-more-data"><?php _e('Add More Data', 'wc-chatbot'); ?></a>
                        </div>

                        <!-- Right content: sub-tab panels -->
                        <div class="wccb-subtab-content">

                            <!-- SUB-TAB 1: Main Company Training -->
                            <div class="wccb-subtab-panel" id="wccb-subtab-main-training" data-subtab="main-training">
                                <div class="wccb-card">
                                    <h2><?php _e('Main Company Training', 'wc-chatbot'); ?></h2>
                                    <p class="description" style="margin:-5px 0 15px 0;color:#666;"><?php _e('Configure how the AI chatbot behaves for your store.', 'wc-chatbot'); ?></p>

                                    <table class="form-table">
                                        <tr>
                                            <th scope="row"><label for="bot_language"><?php _e('Language', 'wc-chatbot'); ?></label></th>
                                            <td>
                                                <select id="bot_language" name="wccb_settings[bot_language]" class="wccb-select2" style="width:100%;max-width:400px;">
                                                    <option value=""><?php _e('-- Select Language --', 'wc-chatbot'); ?></option>
                                                    <?php
                                                    $languages = wccb_get_languages_list();
                                                    foreach ($languages as $name) {
                                                        printf('<option value="%s" %s>%s</option>', esc_attr($name), selected($bot_language, $name, false), esc_html($name));
                                                    }
                                                    ?>
                                                </select>
                                                <p class="description"><?php _e('The language the chatbot will use to respond.', 'wc-chatbot'); ?></p>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th scope="row"><label for="company_name"><?php _e('Company Name', 'wc-chatbot'); ?></label></th>
                                            <td>
                                                <input type="text" id="company_name" name="wccb_settings[company_name]" value="<?php echo esc_attr($company_name); ?>" class="regular-text" placeholder="<?php esc_attr_e('e.g. Acme Corporation S.A.', 'wc-chatbot'); ?>">
                                                <p class="description"><?php _e('Your legal company name.', 'wc-chatbot'); ?></p>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th scope="row"><label for="brand_name"><?php _e('Brand Name', 'wc-chatbot'); ?></label></th>
                                            <td>
                                                <input type="text" id="brand_name" name="wccb_settings[brand_name]" value="<?php echo esc_attr($brand_name); ?>" class="regular-text" placeholder="<?php esc_attr_e('e.g. Acme Store', 'wc-chatbot'); ?>">
                                                <p class="description"><?php _e('Your brand or trade name.', 'wc-chatbot'); ?></p>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th scope="row"><label for="chatbot_objective"><?php _e('Chatbot Objective', 'wc-chatbot'); ?></label></th>
                                            <td>
                                                <input type="text" id="chatbot_objective" name="wccb_settings[chatbot_objective]" value="<?php echo esc_attr($chatbot_objective); ?>" class="large-text" placeholder="<?php esc_attr_e('e.g. Help customers find products and close sales', 'wc-chatbot'); ?>">
                                                <p class="description"><?php _e('The main goal of the chatbot.', 'wc-chatbot'); ?></p>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th scope="row"><label for="custom_prompt"><?php _e('Custom Prompt', 'wc-chatbot'); ?></label></th>
                                            <td>
                                                <textarea id="custom_prompt" name="wccb_settings[custom_prompt]" rows="8" class="large-text" placeholder="<?php esc_attr_e('Additional instructions for the AI chatbot...', 'wc-chatbot'); ?>"><?php echo esc_textarea($custom_prompt); ?></textarea>
                                                <p class="description"><?php _e('Custom instructions the AI will follow. E.g., mention promotions, return policies, tone of voice.', 'wc-chatbot'); ?></p>
                                            </td>
                                        </tr>
                                    </table>
                                </div>
                            </div>

                            <!-- SUB-TAB 2: Generate Prompt with AI (inline questionnaire) -->
                            <div class="wccb-subtab-panel" id="wccb-subtab-generate-prompt" data-subtab="generate-prompt" style="display:none;">
                                <div class="wccb-card">
                                    <h2><?php _e('Generate Prompt with AI', 'wc-chatbot'); ?></h2>
                                    <p class="description" style="margin:-5px 0 15px 0;color:#666;"><?php _e('Answer these questions about your business. The AI will generate a professional custom prompt based on your answers.', 'wc-chatbot'); ?></p>

                                    <div class="wccb-q-group"><label><?php _e('1. What type of products or services do you sell?', 'wc-chatbot'); ?></label><textarea id="wccb-q1" rows="2" placeholder="<?php esc_attr_e('E.g., solar panels, electronics, organic food...', 'wc-chatbot'); ?>"></textarea></div>
                                    <div class="wccb-q-group"><label><?php _e('2. Who is your target audience?', 'wc-chatbot'); ?></label><input type="text" id="wccb-q2" placeholder="<?php esc_attr_e('E.g., homeowners, young professionals, small businesses...', 'wc-chatbot'); ?>"></div>
                                    <div class="wccb-q-group"><label><?php _e('3. What tone should the chatbot use?', 'wc-chatbot'); ?></label><select id="wccb-q3"><option value="Professional"><?php _e('Professional', 'wc-chatbot'); ?></option><option value="Friendly"><?php _e('Friendly', 'wc-chatbot'); ?></option><option value="Casual"><?php _e('Casual', 'wc-chatbot'); ?></option><option value="Formal"><?php _e('Formal', 'wc-chatbot'); ?></option><option value="Enthusiastic"><?php _e('Enthusiastic', 'wc-chatbot'); ?></option></select></div>
                                    <div class="wccb-q-group"><label><?php _e('4. What are your shipping policies?', 'wc-chatbot'); ?></label><textarea id="wccb-q4" rows="2" placeholder="<?php esc_attr_e('E.g., free shipping over $50, ships within 2 business days...', 'wc-chatbot'); ?>"></textarea></div>
                                    <div class="wccb-q-group"><label><?php _e('5. What is your return or refund policy?', 'wc-chatbot'); ?></label><textarea id="wccb-q5" rows="2" placeholder="<?php esc_attr_e('E.g., 30-day returns, no refunds on sale items...', 'wc-chatbot'); ?>"></textarea></div>
                                    <div class="wccb-q-group"><label><?php _e('6. What payment methods do you accept?', 'wc-chatbot'); ?></label><input type="text" id="wccb-q6" placeholder="<?php esc_attr_e('E.g., credit cards, PayPal, bank transfer...', 'wc-chatbot'); ?>"></div>
                                    <div class="wccb-q-group"><label><?php _e('7. Do you have any current promotions or discounts?', 'wc-chatbot'); ?></label><textarea id="wccb-q7" rows="2" placeholder="<?php esc_attr_e('E.g., 20% off this week, buy 2 get 1 free...', 'wc-chatbot'); ?>"></textarea></div>
                                    <div class="wccb-q-group"><label><?php _e('8. What are your customer service hours?', 'wc-chatbot'); ?></label><input type="text" id="wccb-q8" placeholder="<?php esc_attr_e('E.g., Mon-Fri 9am-6pm, 24/7 support...', 'wc-chatbot'); ?>"></div>
                                    <div class="wccb-q-group"><label><?php _e('9. What makes your brand unique?', 'wc-chatbot'); ?></label><textarea id="wccb-q9" rows="2" placeholder="<?php esc_attr_e('E.g., 10 years of experience, eco-friendly products, best price guarantee...', 'wc-chatbot'); ?>"></textarea></div>
                                    <div class="wccb-q-group"><label><?php _e('10. Should the chatbot recommend related or additional products?', 'wc-chatbot'); ?></label><select id="wccb-q10"><option value="Yes, actively upsell and cross-sell"><?php _e('Yes, actively upsell and cross-sell', 'wc-chatbot'); ?></option><option value="Yes, but only when relevant"><?php _e('Yes, but only when relevant', 'wc-chatbot'); ?></option><option value="No, only answer what is asked"><?php _e('No, only answer what is asked', 'wc-chatbot'); ?></option></select></div>
                                    <div class="wccb-q-group"><label><?php _e('11. Are there any topics the chatbot should avoid?', 'wc-chatbot'); ?></label><textarea id="wccb-q11" rows="2" placeholder="<?php esc_attr_e('E.g., competitor comparisons, politics, pricing negotiations...', 'wc-chatbot'); ?>"></textarea></div>
                                    <div class="wccb-q-group"><label><?php _e('12. What is your business niche? The chatbot will only answer questions related to this topic.', 'wc-chatbot'); ?></label><input type="text" id="wccb-q12" placeholder="<?php esc_attr_e('E.g., solar energy, pet supplies, fashion, home appliances...', 'wc-chatbot'); ?>"></div>
                                    <div class="wccb-q-group">
                                        <label><?php _e('13. What languages should the chatbot respond in? (select one or more)', 'wc-chatbot'); ?></label>
                                        <div class="wccb-lang-checkboxes" id="wccb-q13-langs" style="display:flex;flex-wrap:wrap;gap:6px 18px;margin-top:6px;">
                                            <?php
                                            $promptLangs = [
                                                'Spanish', 'English', 'Portuguese', 'French', 'German',
                                                'Italian', 'Dutch', 'Chinese', 'Japanese', 'Korean',
                                                'Arabic', 'Russian', 'Turkish', 'Hindi', 'Polish',
                                            ];
                                            foreach ($promptLangs as $lang) {
                                                printf(
                                                    '<label style="font-weight:400;cursor:pointer;"><input type="checkbox" class="wccb-q13-cb" value="%s" style="margin-right:4px;">%s</label>',
                                                    esc_attr($lang),
                                                    esc_html($lang)
                                                );
                                            }
                                            ?>
                                        </div>
                                        <input type="text" id="wccb-q13-other" style="margin-top:6px;width:300px;" placeholder="<?php esc_attr_e('Other languages (comma separated)...', 'wc-chatbot'); ?>">
                                    </div>
                                    <div class="wccb-q-group"><label><?php _e('14. Any additional instructions or information?', 'wc-chatbot'); ?></label><textarea id="wccb-q14" rows="2" placeholder="<?php esc_attr_e('Anything else the AI should know about your business...', 'wc-chatbot'); ?>"></textarea></div>

                                    <div style="margin-top: 20px; display: flex; align-items: center; gap: 12px;">
                                        <button type="button" id="wccb-generate-prompt" class="button button-primary"><?php _e('Generate Prompt', 'wc-chatbot'); ?></button>
                                        <span id="wccb-generate-result" class="wccb-result" style="display:none;"></span>
                                    </div>
                                </div>
                            </div>

                            <!-- SUB-TAB 3: Add More Data -->
                            <div class="wccb-subtab-panel" id="wccb-subtab-add-more-data" data-subtab="add-more-data" style="display:none;">
                                <div class="wccb-card">
                                    <h2><?php _e('Add More Data', 'wc-chatbot'); ?></h2>
                                    <p class="description" style="margin:-5px 0 15px 0;color:#666;"><?php _e('Add custom training data. Each entry will be included in the chatbot\'s knowledge base. Use this to add FAQs, policies, or any information missing from conversations.', 'wc-chatbot'); ?></p>

                                    <div id="wccb-training-entries">
                                        <?php foreach ($additional_training as $idx => $entry): ?>
                                            <div class="wccb-training-entry" data-index="<?php echo $idx; ?>">
                                                <div class="wccb-training-entry-header">
                                                    <span class="wccb-training-entry-number">#<?php echo $idx + 1; ?></span>
                                                    <button type="button" class="button button-link-delete wccb-remove-entry"><?php _e('Remove', 'wc-chatbot'); ?></button>
                                                </div>
                                                <div class="wccb-training-entry-fields">
                                                    <label><?php _e('Topic', 'wc-chatbot'); ?></label>
                                                    <input type="text" name="wccb_settings[additional_training][<?php echo $idx; ?>][topic]" value="<?php echo esc_attr($entry['topic'] ?? ''); ?>" class="regular-text" placeholder="<?php esc_attr_e('e.g. Return Policy, Warranty Info, Store Hours...', 'wc-chatbot'); ?>">
                                                    <label><?php _e('Content', 'wc-chatbot'); ?></label>
                                                    <textarea name="wccb_settings[additional_training][<?php echo $idx; ?>][content]" rows="4" class="large-text" placeholder="<?php esc_attr_e('Enter the training information here...', 'wc-chatbot'); ?>"><?php echo esc_textarea($entry['content'] ?? ''); ?></textarea>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>

                                    <button type="button" id="wccb-add-training-entry" class="button button-secondary" style="margin-top: 12px;">+ <?php _e('Add Entry', 'wc-chatbot'); ?></button>
                                </div>
                            </div>

                        </div><!-- /.wccb-subtab-content -->
                    </div><!-- /.wccb-subtab-layout -->

                    <?php submit_button(); ?>
                </div>

                <!-- TAB: Widget Settings -->
                <div class="wccb-tab-panel" id="wccb-tab-widget-settings" data-tab="widget-settings" style="display:none;">
                    <div class="wccb-card">
                        <h2><?php _e('General', 'wc-chatbot'); ?></h2>
                        <table class="form-table">
                            <tr>
                                <th scope="row"><?php _e('Enable Widget', 'wc-chatbot'); ?></th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="wccb_settings[widget_enabled]" value="1"
                                            <?php checked($widget_enabled, true); ?>>
                                        <?php _e('Show chat widget on the frontend', 'wc-chatbot'); ?>
                                    </label>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><label for="widget_bot_name"><?php _e('Bot Name', 'wc-chatbot'); ?></label></th>
                                <td>
                                    <input type="text" id="widget_bot_name" name="wccb_settings[widget_bot_name]" value="<?php echo esc_attr($w_bot_name); ?>" class="regular-text" placeholder="<?php esc_attr_e('e.g. Sales Assistant, Support Agent...', 'wc-chatbot'); ?>">
                                    <p class="description"><?php _e('The name displayed in the chat window header. Leave empty to use the default.', 'wc-chatbot'); ?></p>
                                </td>
                            </tr>
                        </table>
                    </div>

                    <div class="wccb-card">
                        <h2><?php _e('Icon Style', 'wc-chatbot'); ?></h2>
                        <table class="form-table">
                            <tr>
                                <th scope="row"><?php _e('Icon Style', 'wc-chatbot'); ?></th>
                                <td>
                                    <input type="hidden" name="wccb_settings[widget_icon_style]" id="widget_icon_style" value="<?php echo esc_attr($w_icon_style); ?>">
                                    <div class="wccb-icon-grid">
                                        <div class="wccb-icon-option<?php echo $w_icon_style === 'chat' ? ' active' : ''; ?>" data-value="chat">
                                            <div class="wccb-icon-preview">
                                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                    <path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"></path>
                                                </svg>
                                            </div>
                                            <span class="wccb-icon-label"><?php _e('Chat Bubble', 'wc-chatbot'); ?></span>
                                        </div>
                                        <div class="wccb-icon-option<?php echo $w_icon_style === 'headset' ? ' active' : ''; ?>" data-value="headset">
                                            <div class="wccb-icon-preview">
                                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                    <path d="M3 18v-6a9 9 0 0 1 18 0v6"></path>
                                                    <path d="M21 19a2 2 0 0 1-2 2h-1a2 2 0 0 1-2-2v-3a2 2 0 0 1 2-2h3zM3 19a2 2 0 0 0 2 2h1a2 2 0 0 0 2-2v-3a2 2 0 0 0-2-2H3z"></path>
                                                </svg>
                                            </div>
                                            <span class="wccb-icon-label"><?php _e('Headset', 'wc-chatbot'); ?></span>
                                        </div>
                                        <div class="wccb-icon-option<?php echo $w_icon_style === 'shopping' ? ' active' : ''; ?>" data-value="shopping">
                                            <div class="wccb-icon-preview">
                                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                    <path d="M6 2L3 6v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2V6l-3-4z"></path>
                                                    <line x1="3" y1="6" x2="21" y2="6"></line>
                                                    <path d="M16 10a4 4 0 0 1-8 0"></path>
                                                </svg>
                                            </div>
                                            <span class="wccb-icon-label"><?php _e('Shopping Bag', 'wc-chatbot'); ?></span>
                                        </div>
                                        <div class="wccb-icon-option<?php echo $w_icon_style === 'robot' ? ' active' : ''; ?>" data-value="robot">
                                            <div class="wccb-icon-preview">
                                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                    <rect x="3" y="8" width="18" height="12" rx="2"></rect>
                                                    <circle cx="9" cy="14" r="1.5" fill="currentColor"></circle>
                                                    <circle cx="15" cy="14" r="1.5" fill="currentColor"></circle>
                                                    <path d="M12 2v4"></path>
                                                    <circle cx="12" cy="2" r="1"></circle>
                                                </svg>
                                            </div>
                                            <span class="wccb-icon-label"><?php _e('Robot', 'wc-chatbot'); ?></span>
                                        </div>
                                        <div class="wccb-icon-option<?php echo $w_icon_style === 'support' ? ' active' : ''; ?>" data-value="support">
                                            <div class="wccb-icon-preview">
                                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                    <path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"></path>
                                                    <circle cx="12" cy="7" r="4"></circle>
                                                    <path d="M5.5 10.5L3 12v3h3l2-2"></path>
                                                    <path d="M18.5 10.5L21 12v3h-3l-2-2"></path>
                                                </svg>
                                            </div>
                                            <span class="wccb-icon-label"><?php _e('Support Agent', 'wc-chatbot'); ?></span>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                        </table>
                    </div>

                    <div class="wccb-card">
                        <h2><?php _e('Desktop Settings', 'wc-chatbot'); ?></h2>

                        <h3 style="margin:0 0 5px 0;padding-top:0;"><?php _e('Floating Icon', 'wc-chatbot'); ?></h3>
                        <table class="form-table" style="margin-top:0;">
                            <tr>
                                <th scope="row"><?php _e('Icon Size (px)', 'wc-chatbot'); ?></th>
                                <td>
                                    <input type="number" name="wccb_settings[widget_icon_size]" value="<?php echo esc_attr($w_icon_size); ?>" min="40" max="80" step="2" style="width:80px;">
                                    <p class="description"><?php _e('40-80px. Default: 60px.', 'wc-chatbot'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Position', 'wc-chatbot'); ?></th>
                                <td>
                                    <select name="wccb_settings[widget_position]">
                                        <option value="bottom-right" <?php selected($w_position, 'bottom-right'); ?>><?php _e('Bottom Right', 'wc-chatbot'); ?></option>
                                        <option value="bottom-left" <?php selected($w_position, 'bottom-left'); ?>><?php _e('Bottom Left', 'wc-chatbot'); ?></option>
                                        <option value="top-right" <?php selected($w_position, 'top-right'); ?>><?php _e('Top Right', 'wc-chatbot'); ?></option>
                                        <option value="top-left" <?php selected($w_position, 'top-left'); ?>><?php _e('Top Left', 'wc-chatbot'); ?></option>
                                    </select>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Offset X / Y (px)', 'wc-chatbot'); ?></th>
                                <td>
                                    <input type="number" name="wccb_settings[widget_offset_x]" value="<?php echo esc_attr($w_offset_x); ?>" min="0" max="200" step="1" style="width:80px;">
                                    <span>&times;</span>
                                    <input type="number" name="wccb_settings[widget_offset_y]" value="<?php echo esc_attr($w_offset_y); ?>" min="0" max="200" step="1" style="width:80px;">
                                    <p class="description"><?php _e('Horizontal and vertical distance from the edge.', 'wc-chatbot'); ?></p>
                                </td>
                            </tr>
                        </table>

                        <h3 style="margin:10px 0 5px 0;"><?php _e('Chat Window', 'wc-chatbot'); ?></h3>
                        <table class="form-table" style="margin-top:0;">
                            <tr>
                                <th scope="row"><?php _e('Width / Height (px)', 'wc-chatbot'); ?></th>
                                <td>
                                    <input type="number" name="wccb_settings[widget_chat_width]" value="<?php echo esc_attr($w_chat_width); ?>" min="300" max="500" step="10" style="width:80px;">
                                    <span>&times;</span>
                                    <input type="number" name="wccb_settings[widget_chat_height]" value="<?php echo esc_attr($w_chat_height); ?>" min="400" max="700" step="10" style="width:80px;">
                                    <p class="description"><?php _e('Width: 300-500px. Height: 400-700px.', 'wc-chatbot'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Offset X / Y (px)', 'wc-chatbot'); ?></th>
                                <td>
                                    <input type="number" name="wccb_settings[widget_chat_offset_x]" value="<?php echo esc_attr($w_chat_offset_x); ?>" min="-200" max="200" step="1" style="width:80px;">
                                    <span>&times;</span>
                                    <input type="number" name="wccb_settings[widget_chat_offset_y]" value="<?php echo esc_attr($w_chat_offset_y); ?>" min="0" max="100" step="1" style="width:80px;">
                                    <p class="description"><?php _e('Horizontal and vertical offset of the chat window from the icon. Default: 0 x 15.', 'wc-chatbot'); ?></p>
                                </td>
                            </tr>
                        </table>
                    </div>

                    <div class="wccb-card">
                        <h2><?php _e('Mobile Settings', 'wc-chatbot'); ?></h2>
                        <p class="description" style="margin:-5px 0 10px 0;color:#666;"><?php _e('These settings apply on screens narrower than 768px. Set Width/Height to 0 to use full screen.', 'wc-chatbot'); ?></p>

                        <h3 style="margin:0 0 5px 0;padding-top:0;"><?php _e('Floating Icon', 'wc-chatbot'); ?></h3>
                        <table class="form-table" style="margin-top:0;">
                            <tr>
                                <th scope="row"><?php _e('Icon Size (px)', 'wc-chatbot'); ?></th>
                                <td>
                                    <input type="number" name="wccb_settings[widget_mobile_icon_size]" value="<?php echo esc_attr($wm_icon_size); ?>" min="36" max="70" step="2" style="width:80px;">
                                    <p class="description"><?php _e('36-70px. Default: 50px.', 'wc-chatbot'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Position', 'wc-chatbot'); ?></th>
                                <td>
                                    <select name="wccb_settings[widget_mobile_position]">
                                        <option value="bottom-right" <?php selected($wm_position, 'bottom-right'); ?>><?php _e('Bottom Right', 'wc-chatbot'); ?></option>
                                        <option value="bottom-left" <?php selected($wm_position, 'bottom-left'); ?>><?php _e('Bottom Left', 'wc-chatbot'); ?></option>
                                        <option value="top-right" <?php selected($wm_position, 'top-right'); ?>><?php _e('Top Right', 'wc-chatbot'); ?></option>
                                        <option value="top-left" <?php selected($wm_position, 'top-left'); ?>><?php _e('Top Left', 'wc-chatbot'); ?></option>
                                    </select>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Offset X / Y (px)', 'wc-chatbot'); ?></th>
                                <td>
                                    <input type="number" name="wccb_settings[widget_mobile_offset_x]" value="<?php echo esc_attr($wm_offset_x); ?>" min="0" max="100" step="1" style="width:80px;">
                                    <span>&times;</span>
                                    <input type="number" name="wccb_settings[widget_mobile_offset_y]" value="<?php echo esc_attr($wm_offset_y); ?>" min="0" max="100" step="1" style="width:80px;">
                                    <p class="description"><?php _e('Horizontal and vertical distance from the edge.', 'wc-chatbot'); ?></p>
                                </td>
                            </tr>
                        </table>

                        <h3 style="margin:10px 0 5px 0;"><?php _e('Chat Window', 'wc-chatbot'); ?></h3>
                        <table class="form-table" style="margin-top:0;">
                            <tr>
                                <th scope="row"><?php _e('Width / Height (px)', 'wc-chatbot'); ?></th>
                                <td>
                                    <input type="number" name="wccb_settings[widget_mobile_chat_width]" value="<?php echo esc_attr($wm_chat_width); ?>" min="0" max="500" step="10" style="width:80px;">
                                    <span>&times;</span>
                                    <input type="number" name="wccb_settings[widget_mobile_chat_height]" value="<?php echo esc_attr($wm_chat_height); ?>" min="0" max="700" step="10" style="width:80px;">
                                    <p class="description"><?php _e('Set to 0 for full screen. Default: 0 x 0 (full screen).', 'wc-chatbot'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Offset X / Y (px)', 'wc-chatbot'); ?></th>
                                <td>
                                    <input type="number" name="wccb_settings[widget_mobile_chat_offset_x]" value="<?php echo esc_attr($wm_chat_offset_x); ?>" min="-100" max="100" step="1" style="width:80px;">
                                    <span>&times;</span>
                                    <input type="number" name="wccb_settings[widget_mobile_chat_offset_y]" value="<?php echo esc_attr($wm_chat_offset_y); ?>" min="0" max="50" step="1" style="width:80px;">
                                    <p class="description"><?php _e('Default: 0 x 10.', 'wc-chatbot'); ?></p>
                                </td>
                            </tr>
                        </table>
                    </div>

                    <div class="wccb-card">
                        <h2><?php _e('Colors', 'wc-chatbot'); ?></h2>

                        <table class="form-table">
                            <tr>
                                <th scope="row"><?php _e('Primary Color', 'wc-chatbot'); ?></th>
                                <td>
                                    <div class="wccb-color-field">
                                        <input type="color" name="wccb_settings[widget_primary_color]" value="<?php echo esc_attr($w_primary_color); ?>">
                                        <code class="wccb-color-hex"><?php echo esc_html($w_primary_color); ?></code>
                                    </div>
                                    <p class="description"><?php _e('Header background, send button, and toggle button.', 'wc-chatbot'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Bot Bubble Color', 'wc-chatbot'); ?></th>
                                <td>
                                    <div class="wccb-color-field">
                                        <input type="color" name="wccb_settings[widget_bot_bubble_color]" value="<?php echo esc_attr($w_bot_bubble_color); ?>">
                                        <code class="wccb-color-hex"><?php echo esc_html($w_bot_bubble_color); ?></code>
                                    </div>
                                    <p class="description"><?php _e('Background color of bot message bubbles.', 'wc-chatbot'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Header Text Color', 'wc-chatbot'); ?></th>
                                <td>
                                    <div class="wccb-color-field">
                                        <input type="color" name="wccb_settings[widget_header_text_color]" value="<?php echo esc_attr($w_header_text_color); ?>">
                                        <code class="wccb-color-hex"><?php echo esc_html($w_header_text_color); ?></code>
                                    </div>
                                    <p class="description"><?php _e('Text color in the chat header.', 'wc-chatbot'); ?></p>
                                </td>
                            </tr>
                        </table>
                    </div>

                    <?php submit_button(); ?>
                </div>
            </form>

            <!-- TAB: Chat History (outside form, loaded via AJAX) -->
            <div class="wccb-tab-panel" id="wccb-tab-chat-history" data-tab="chat-history" style="display:none;">
                <div class="wccb-card">
                    <h2><?php _e('Chat History', 'wc-chatbot'); ?></h2>
                    <p class="description" style="margin:-5px 0 15px 0;color:#666;"><?php _e('Recent conversations from your store visitors.', 'wc-chatbot'); ?></p>

                    <div id="wccb-chat-history-loading" style="text-align:center;padding:40px;">
                        <span class="spinner is-active" style="float:none;"></span>
                    </div>

                    <div id="wccb-chat-history-content" style="display:none;">
                        <table class="wp-list-table widefat fixed striped" id="wccb-chat-table">
                            <thead>
                                <tr>
                                    <th style="width:160px;"><?php _e('Date', 'wc-chatbot'); ?></th>
                                    <th style="width:130px;"><?php _e('Visitor IP', 'wc-chatbot'); ?></th>
                                    <th><?php _e('Session', 'wc-chatbot'); ?></th>
                                    <th style="width:90px;"><?php _e('Messages', 'wc-chatbot'); ?></th>
                                    <th style="width:80px;"><?php _e('Actions', 'wc-chatbot'); ?></th>
                                </tr>
                            </thead>
                            <tbody id="wccb-chat-history-body"></tbody>
                        </table>

                        <div id="wccb-chat-pagination" class="tablenav bottom" style="margin-top:10px;">
                            <div class="tablenav-pages" id="wccb-chat-pages"></div>
                        </div>
                    </div>

                    <div id="wccb-chat-history-empty" style="display:none;text-align:center;padding:40px;color:#666;">
                        <?php _e('No chat logs found.', 'wc-chatbot'); ?>
                    </div>

                    <div id="wccb-chat-history-error" style="display:none;" class="wccb-result error"></div>
                </div>
            </div>
        </div>

        <!-- Chat Detail Modal -->
        <div id="wccb-chat-detail-modal" class="wccb-modal-overlay" style="display:none;">
            <div class="wccb-modal">
                <div class="wccb-modal-header">
                    <h2><?php _e('Conversation Details', 'wc-chatbot'); ?></h2>
                    <button type="button" class="wccb-modal-close">&times;</button>
                </div>
                <div class="wccb-modal-body">
                    <div class="wccb-chat-detail-meta" id="wccb-chat-detail-meta"></div>
                    <div class="wccb-chat-detail-messages" id="wccb-chat-detail-messages"></div>
                </div>
                <div class="wccb-modal-footer">
                    <button type="button" class="button wccb-modal-close"><?php _e('Close', 'wc-chatbot'); ?></button>
                </div>
            </div>
        </div>

        <!-- Questionnaire modal removed — now inline in Generate Prompt with AI sub-tab -->
        <?php
    }

    /**
     * AJAX: Test connection
     */
    public function ajax_test_connection() {
        check_ajax_referer('wccb_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied', 'wc-chatbot')]);
        }

        $hub_url = WC_Chatbot::get_option('hub_url', 'https://woocommercechatbot.com');
        $license_key = isset($_POST['license_key']) ? sanitize_text_field($_POST['license_key']) : '';

        $result = wccb()->api->test_handshake($hub_url, $license_key);

        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }

        $store = $result['store'] ?? [];

        // Update local options with fresh Hub data
        $chats_used = $store['chats_used'] ?? 0;
        $monthly_limit = $store['monthly_limit'] ?? 50;
        WC_Chatbot::update_option('messages_used', $chats_used);
        WC_Chatbot::update_option('monthly_limit', $monthly_limit);
        WC_Chatbot::update_option('store_status', $store['status'] ?? 'unknown');

        wp_send_json_success([
            'message' => __('Connection successful!', 'wc-chatbot'),
            'store' => $store,
            'status' => $store['status'] ?? 'unknown',
            'expires_at' => $store['expires_at'] ?? null,
            'monthly_limit' => $monthly_limit,
            'chats_used' => $chats_used,
        ]);
    }

    /**
     * AJAX: Silent ping to send plugin version to hub
     */
    public function ajax_ping_version() {
        check_ajax_referer('wccb_admin_nonce', 'nonce');

        $hub_url = WC_Chatbot::get_option('hub_url');
        $license_key = WC_Chatbot::get_option('license_key');

        if (empty($hub_url) || empty($license_key)) {
            wp_send_json_success(['pinged' => false]);
        }

        wccb()->api->handshake();
        wp_send_json_success(['pinged' => true]);
    }

    /**
     * AJAX: Sync products now
     */
    public function ajax_sync_now() {
        check_ajax_referer('wccb_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied', 'wc-chatbot')]);
        }

        $result = wccb()->sync->sync_products();

        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }

        wp_send_json_success([
            'message' => sprintf(
                __('Synced %d products successfully!', 'wc-chatbot'),
                $result['products_count'] ?? 0
            ),
        ]);
    }

    /**
     * AJAX: Batched sync with progress
     */
    public function ajax_sync_batch() {
        check_ajax_referer('wccb_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied', 'wc-chatbot')]);
        }

        $step = sanitize_text_field($_POST['step'] ?? '');

        if ($step === 'count') {
            $total = wccb()->sync->get_products_count();
            wp_send_json_success(['total' => $total]);
        }

        if ($step === 'batch') {
            $offset = intval($_POST['offset'] ?? 0);
            $limit = intval($_POST['limit'] ?? 100);
            $isFirst = !empty($_POST['is_first']);
            $isLast = !empty($_POST['is_last']);
            $total = intval($_POST['total'] ?? 0);

            $products = wccb()->sync->get_products_batch($offset, $limit);
            $result = wccb()->api->sync_products_batch($products, $isFirst, $isLast, $total);

            if (is_wp_error($result)) {
                wp_send_json_error(['message' => $result->get_error_message()]);
            }

            if ($isLast) {
                WC_Chatbot::update_option('last_sync', current_time('mysql'));
                WC_Chatbot::update_option('products_synced', $total);
            }

            wp_send_json_success(['synced' => count($products)]);
        }

        wp_send_json_error(['message' => 'Invalid step']);
    }

    /**
     * AJAX: Generate prompt from questionnaire answers
     */
    public function ajax_generate_prompt() {
        check_ajax_referer('wccb_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied', 'wc-chatbot')]);
        }

        $answers = [];
        for ($i = 1; $i <= 14; $i++) {
            $answers['q' . $i] = sanitize_textarea_field($_POST['q' . $i] ?? '');
        }

        $result = wccb()->api->generate_prompt($answers);

        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }

        wp_send_json_success(['prompt' => $result['prompt'] ?? '']);
    }

    /**
     * AJAX: Fetch chat history from Hub
     */
    public function ajax_fetch_chat_history() {
        check_ajax_referer('wccb_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied', 'wc-chatbot')]);
        }

        $page = isset($_POST['page']) ? intval($_POST['page']) : 1;
        $per_page = isset($_POST['per_page']) ? intval($_POST['per_page']) : 20;

        $result = wccb()->api->get_chat_logs($page, $per_page);

        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }

        // Enrich assistant messages: decode HTML entities + add product cards from local WooCommerce
        if (!empty($result['logs']) && class_exists('WooCommerce')) {
            $result['logs'] = $this->enrich_chat_history($result['logs']);
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Fetch single chat log (for live modal polling)
     */
    public function ajax_fetch_single_chat() {
        check_ajax_referer('wccb_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $id = isset($_POST['chat_log_id']) ? intval($_POST['chat_log_id']) : 0;
        if (!$id) {
            wp_send_json_error(['message' => 'Invalid ID']);
        }

        $result = wccb()->api->get_single_chat_log($id);
        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }

        // Enrich with local product data
        if (!empty($result['log']) && class_exists('WooCommerce')) {
            $enriched = $this->enrich_chat_history([$result['log']]);
            $result['log'] = $enriched[0];
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Delete a chat conversation
     */
    public function ajax_delete_chat() {
        check_ajax_referer('wccb_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied', 'wc-chatbot')]);
        }

        $chat_log_id = isset($_POST['chat_log_id']) ? intval($_POST['chat_log_id']) : 0;
        if (!$chat_log_id) {
            wp_send_json_error(['message' => __('Invalid chat ID', 'wc-chatbot')]);
        }

        $result = wccb()->api->delete_chat($chat_log_id);

        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }

        wp_send_json_success(['message' => __('Chat deleted', 'wc-chatbot')]);
    }

    /**
     * AJAX: Get quick access URL for admin panel
     */
    public function ajax_quick_access() {
        check_ajax_referer('wccb_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied', 'wc-chatbot')]);
        }

        $result = wccb()->api->get_quick_access_url();

        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }

        wp_send_json_success($result);
    }

    /**
     * Enrich chat history: decode HTML entities in content and always extract
     * product cards from local WooCommerce data for assistant messages.
     */
    private function enrich_chat_history($logs) {
        global $wpdb;

        // Load all published product titles once (fast single query)
        $all_products = $wpdb->get_results(
            "SELECT ID, post_title FROM {$wpdb->posts}
             WHERE post_type = 'product' AND post_status = 'publish'"
        );

        error_log('WCCB enrich_history: ' . count($logs) . ' logs, ' . (is_array($all_products) ? count($all_products) : 0) . ' products in DB');

        foreach ($logs as &$log) {
            if (empty($log['history'])) continue;

            foreach ($log['history'] as &$msg) {
                if (($msg['role'] ?? '') !== 'assistant') continue;

                // Decode HTML entities in content (fixes &#36; → $)
                if (!empty($msg['content'])) {
                    $msg['content'] = html_entity_decode($msg['content'], ENT_QUOTES | ENT_HTML5, 'UTF-8');
                }

                $reply = $msg['content'] ?? '';
                if (empty($reply) || empty($all_products)) continue;

                // Always extract products locally (replaces any Hub-stored products with full local data)
                $suggested = [];
                foreach ($all_products as $row) {
                    if (count($suggested) >= 5) break;

                    $title = html_entity_decode($row->post_title, ENT_QUOTES | ENT_HTML5, 'UTF-8');
                    if (empty($title) || mb_strlen($title) < 4) continue;

                    $matched = false;

                    // Direct match (case-insensitive, multibyte-safe)
                    if (mb_stripos($reply, $title) !== false) {
                        $matched = true;
                    }

                    // Normalized match (strip accents)
                    if (!$matched) {
                        $title_clean = mb_strtolower(remove_accents($title));
                        $reply_clean = mb_strtolower(remove_accents($reply));
                        if (mb_strpos($reply_clean, $title_clean) !== false) {
                            $matched = true;
                        }
                    }

                    if ($matched) {
                        $product = wc_get_product($row->ID);
                        if ($product) {
                            $image_id = $product->get_image_id();
                            $image_url = $image_id ? wp_get_attachment_image_url($image_id, 'woocommerce_thumbnail') : '';
                            $suggested[] = [
                                'id'    => $product->get_id(),
                                'name'  => html_entity_decode($product->get_name(), ENT_QUOTES | ENT_HTML5, 'UTF-8'),
                                'price' => $product->get_price(),
                                'url'   => get_permalink($product->get_id()),
                                'image' => $image_url ?: '',
                            ];
                            error_log('WCCB enrich_history: matched "' . $title . '" → ID ' . $row->ID);
                        }
                    }
                }

                if (!empty($suggested)) {
                    $msg['products'] = $suggested;
                    error_log('WCCB enrich_history: set ' . count($suggested) . ' products on message');
                }
            }
            unset($msg);
        }
        unset($log);

        return $logs;
    }
}

/**
 * Get list of languages for the bot training selector
 */
function wccb_get_languages_list() {
    return [
        'Spanish / Español',
        'English',
        'Portuguese / Português',
        'French / Français',
        'German / Deutsch',
        'Italian / Italiano',
        'Dutch / Nederlands',
        'Russian / Русский',
        'Chinese / 中文',
        'Japanese / 日本語',
        'Korean / 한국어',
        'Arabic / العربية',
        'Hindi / हिन्दी',
        'Bengali / বাংলা',
        'Turkish / Türkçe',
        'Polish / Polski',
        'Ukrainian / Українська',
        'Vietnamese / Tiếng Việt',
        'Thai / ไทย',
        'Swedish / Svenska',
        'Danish / Dansk',
        'Finnish / Suomi',
        'Norwegian / Norsk',
        'Czech / Čeština',
        'Greek / Ελληνικά',
        'Hebrew / עברית',
        'Indonesian / Bahasa Indonesia',
        'Malay / Bahasa Melayu',
        'Romanian / Română',
        'Hungarian / Magyar',
        'Bulgarian / Български',
        'Croatian / Hrvatski',
        'Slovak / Slovenčina',
        'Slovenian / Slovenščina',
        'Lithuanian / Lietuvių',
        'Latvian / Latviešu',
        'Estonian / Eesti',
        'Serbian / Srpski',
        'Catalan / Català',
        'Galician / Galego',
        'Basque / Euskara',
        'Filipino / Tagalog',
        'Swahili',
        'Afrikaans',
        'Amharic / አማርኛ',
        'Persian / فارسی',
        'Urdu / اردو',
        'Tamil / தமிழ்',
        'Telugu / తెలుగు',
        'Malayalam / മലയാളം',
        'Kannada / ಕನ್ನಡ',
        'Marathi / मराठी',
        'Gujarati / ગુજરાતી',
        'Punjabi / ਪੰਜਾਬੀ',
        'Burmese / မြန်မာ',
        'Khmer / ខ្មែរ',
        'Lao / ລາວ',
        'Georgian / ქართული',
        'Armenian / Հայերեն',
        'Azerbaijani / Azərbaycan',
        'Uzbek / Oʻzbek',
        'Kazakh / Қазақ',
        'Mongolian / Монгол',
        'Nepali / नेपाली',
        'Sinhala / සිංහල',
        'Icelandic / Íslenska',
        'Albanian / Shqip',
        'Macedonian / Македонски',
        'Bosnian / Bosanski',
    ];
}
