/**
 * WC Chatbot Widget JavaScript
 */

(function() {
    'use strict';

    // DOM Elements
    var widget = document.getElementById('wccb-widget');
    var toggle = document.getElementById('wccb-toggle');
    var chat = document.getElementById('wccb-chat');
    var messages = document.getElementById('wccb-messages');
    var form = document.getElementById('wccb-form');
    var input = document.getElementById('wccb-input');

    // State
    var isOpen = false;
    var isLoading = false;
    var sessionId = wccb_widget.session_id;
    var lastMessageTime = 0;
    var THROTTLE_MS = 3000;
    var humanPollTimer = null;
    var humanPollInFlight = false;
    var waitingForHumanReply = false;
    var heartbeatTimer = null;
    var heartbeatStarted = false;

    // Storage keys
    var STORAGE_OPEN = 'wccb_open';
    var STORAGE_MESSAGES = 'wccb_messages';

    // Design settings
    var design = wccb_widget.design || {};
    var MOBILE_BREAKPOINT = 768;
    var currentMode = ''; // track to avoid redundant re-applies

    // Colors (shared, apply once)
    var primaryColor = design.primary_color || '#4F46E5';
    widget.style.setProperty('--wccb-primary', primaryColor);
    widget.style.setProperty('--wccb-primary-dark', darkenColor(primaryColor, 15));
    if (design.bot_bubble_color) widget.style.setProperty('--wccb-bot-bg', design.bot_bubble_color);
    if (design.header_text_color) widget.style.setProperty('--wccb-header-text', design.header_text_color);

    // Icon style (shared, apply once)
    var iconStyle = design.icon_style || 'chat';
    var allIcons = toggle.querySelectorAll('.wccb-icon');
    for (var i = 0; i < allIcons.length; i++) {
        allIcons[i].classList.remove('active');
    }
    var activeIcon = toggle.querySelector('.wccb-icon-' + iconStyle);
    if (activeIcon) {
        activeIcon.classList.add('active');
    } else {
        var fallback = toggle.querySelector('.wccb-icon-chat');
        if (fallback) fallback.classList.add('active');
    }

    // Apply layout settings based on device
    function applyLayout() {
        var isMobile = window.innerWidth <= MOBILE_BREAKPOINT;
        var mode = isMobile ? 'mobile' : 'desktop';
        if (mode === currentMode) return;
        currentMode = mode;

        var iconSize, pos, ox, oy, chatW, chatH, chatOffsetX, chatOffsetY;

        if (isMobile) {
            iconSize = design.mobile_icon_size || 50;
            pos = design.mobile_position || design.position || 'bottom-right';
            ox = (typeof design.mobile_offset_x !== 'undefined' ? design.mobile_offset_x : 15) + 'px';
            oy = (typeof design.mobile_offset_y !== 'undefined' ? design.mobile_offset_y : 15) + 'px';
            chatW = design.mobile_chat_width || 0;
            chatH = design.mobile_chat_height || 0;
            chatOffsetX = typeof design.mobile_chat_offset_x !== 'undefined' ? design.mobile_chat_offset_x : 0;
            chatOffsetY = typeof design.mobile_chat_offset_y !== 'undefined' ? design.mobile_chat_offset_y : 10;
        } else {
            iconSize = design.icon_size || 60;
            pos = design.position || 'bottom-right';
            ox = (design.offset_x || 20) + 'px';
            oy = (design.offset_y || 20) + 'px';
            chatW = design.chat_width || 380;
            chatH = design.chat_height || 500;
            chatOffsetX = typeof design.chat_offset_x !== 'undefined' ? design.chat_offset_x : 0;
            chatOffsetY = typeof design.chat_offset_y !== 'undefined' ? design.chat_offset_y : 15;
        }

        // Icon dimensions
        widget.style.setProperty('--wccb-icon-size', iconSize + 'px');
        widget.style.setProperty('--wccb-icon-svg', Math.round(iconSize * 0.47) + 'px');

        // Chat window dimensions (0 = full screen on mobile)
        if (chatW === 0 && isMobile) {
            widget.style.setProperty('--wccb-chat-width', '100vw');
        } else {
            widget.style.setProperty('--wccb-chat-width', chatW + 'px');
        }
        if (chatH === 0 && isMobile) {
            widget.style.setProperty('--wccb-chat-height', 'calc(100vh - ' + (iconSize + 20) + 'px)');
        } else {
            widget.style.setProperty('--wccb-chat-height', chatH + 'px');
        }

        // Widget position
        widget.style.top = 'auto';
        widget.style.bottom = 'auto';
        widget.style.left = 'auto';
        widget.style.right = 'auto';

        if (pos === 'bottom-right') { widget.style.bottom = oy; widget.style.right = ox; }
        else if (pos === 'bottom-left') { widget.style.bottom = oy; widget.style.left = ox; }
        else if (pos === 'top-right') { widget.style.top = oy; widget.style.right = ox; }
        else if (pos === 'top-left') { widget.style.top = oy; widget.style.left = ox; }

        // Chat window alignment with offset
        chat.style.right = chatOffsetX + 'px';
        chat.style.left = 'auto';
        if (pos.indexOf('left') !== -1) {
            chat.style.right = 'auto';
            chat.style.left = chatOffsetX + 'px';
        }
        if (chatW === 0 && isMobile) {
            // Full-width: override left/right to 0
            chat.style.left = '0';
            chat.style.right = '0';
        }
        if (pos.indexOf('top') !== -1) {
            chat.style.bottom = 'auto';
            chat.style.top = (iconSize + chatOffsetY) + 'px';
        } else {
            chat.style.bottom = (iconSize + chatOffsetY) + 'px';
            chat.style.top = 'auto';
        }
        if (chatH === 0 && isMobile && pos.indexOf('top') === -1) {
            chat.style.top = '0';
        }
    }

    // Apply on load
    applyLayout();

    // Re-apply on resize (debounced)
    var resizeTimer;
    window.addEventListener('resize', function() {
        clearTimeout(resizeTimer);
        resizeTimer = setTimeout(applyLayout, 150);
    });

    // Set session cookie
    if (!getCookie('wccb_session')) {
        setCookie('wccb_session', sessionId, 30);
    } else {
        sessionId = getCookie('wccb_session');
    }

    // Restore chat state from sessionStorage
    restoreChatState();

    // Toggle chat
    toggle.addEventListener('click', function() {
        isOpen = !isOpen;
        widget.classList.toggle('open', isOpen);
        saveChatOpen(isOpen);

        if (isOpen) {
            input.focus();
            scrollToBottom();
        }
    });

    // Handle form submit
    form.addEventListener('submit', function(e) {
        e.preventDefault();

        var message = input.value.trim();
        if (!message || isLoading) return;

        // Throttle: minimum 3 seconds between messages
        var now = Date.now();
        if (now - lastMessageTime < THROTTLE_MS) {
            var sendBtn = form.querySelector('.wccb-send');
            if (sendBtn) {
                sendBtn.disabled = true;
                setTimeout(function() { sendBtn.disabled = false; }, THROTTLE_MS - (now - lastMessageTime));
            }
            return;
        }
        lastMessageTime = now;

        // Add user message
        addMessage(message, 'user');
        input.value = '';

        // Show typing indicator
        showTyping();

        // Send to server
        sendMessage(message);
    });

    // Send message to server
    function sendMessage(message) {
        isLoading = true;

        var formData = new FormData();
        formData.append('action', 'wccb_chat');
        formData.append('nonce', wccb_widget.nonce);
        formData.append('message', message);
        formData.append('session_id', sessionId);

        fetch(wccb_widget.ajax_url, {
            method: 'POST',
            body: formData
        })
        .then(function(response) {
            return response.json();
        })
        .then(function(data) {
            if (data.success) {
                // Update session if changed
                if (data.data.session_id) {
                    sessionId = data.data.session_id;
                    setCookie('wccb_session', sessionId, 30);
                }

                // Human mode: hide typing (no bot is writing), start polling for human reply
                if (data.data.human_mode) {
                    hideTyping();
                    waitingForHumanReply = true;
                    startHumanPoll();
                    return;
                }

                hideTyping();
                addMessage(data.data.reply, 'bot', data.data.suggested_products);
            } else {
                hideTyping();
                addMessage(data.data.message || wccb_widget.strings.error, 'bot');
            }
        })
        .catch(function() {
            hideTyping();
            addMessage(wccb_widget.strings.error, 'bot');
        })
        .finally(function() {
            isLoading = false;
        });
    }

    // Fix prices with unwanted decimals (e.g. $4.496,5 → $4.497 when num_decimals=0)
    function cleanPrices(text) {
        var pf = wccb_widget.price_format || {};
        if (!pf.currency_symbol) return text;
        var decimals = parseInt(pf.num_decimals || 0);
        if (decimals > 0) return text;
        var sym = pf.currency_symbol || '$';
        var tSep = pf.thousand_sep || '.';
        var dSep = pf.decimal_sep || ',';
        var escSym = sym.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        var escT = tSep.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        var escD = dSep.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        var re = new RegExp(escSym + '\\s?([\\d' + escT + ']+)' + escD + '(\\d+)', 'g');
        return text.replace(re, function(m, intPart, decPart) {
            var num = parseFloat(intPart.replace(new RegExp(escT, 'g'), '') + '.' + decPart);
            num = Math.round(num);
            var s = num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, tSep);
            return sym + s;
        });
    }

    // Format bot response: markdown-like to HTML
    function formatBotMessage(text) {
        // Clean up prices with wrong decimals before rendering
        text = cleanPrices(text);
        // Escape HTML first
        var escaped = text.replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;');

        // Bold: **text**
        escaped = escaped.replace(/\*\*(.+?)\*\*/g, '<strong>$1</strong>');

        // Markdown links: [text](url)
        escaped = escaped.replace(/\[([^\]]+)\]\((https?:\/\/[^\s)]+)\)/g, '<a href="$2" target="_self">$1</a>');

        // Standalone URLs (not already inside an href)
        escaped = escaped.replace(/(^|[^"=])(https?:\/\/[^\s<),]+)/g, '$1<a href="$2" target="_self">$2</a>');

        // Numbered lists: "1. item" at start of line → list item with number
        // Only match 1-2 digit numbers at start of string or after newline (avoids breaking prices like $75.000)
        escaped = escaped.replace(/(^|\n)(\d{1,2})\.\s+/g, '$1<br><strong>$2.</strong> ');

        // Remove leading <br> if first thing
        escaped = escaped.replace(/^(<br>)+/, '');

        // Line breaks
        escaped = escaped.replace(/\n/g, '<br>');

        return escaped;
    }

    // Format price using WooCommerce settings
    function formatPrice(price) {
        var pf = wccb_widget.price_format || {};
        var symbol = pf.currency_symbol || '$';
        var thousandSep = pf.thousand_sep || ',';
        var decimalSep = pf.decimal_sep || '.';
        var decimals = typeof pf.num_decimals !== 'undefined' ? parseInt(pf.num_decimals) : 2;
        var posFormat = pf.currency_pos || 'left';

        var num = parseFloat(price);
        if (isNaN(num)) return price;

        // Format number with thousand separator and decimal separator
        var fixed = num.toFixed(decimals);
        var parts = fixed.split('.');
        var intPart = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, thousandSep);
        var formatted = decimals > 0 ? intPart + decimalSep + parts[1] : intPart;

        // Apply currency position
        if (posFormat === 'left') return symbol + formatted;
        if (posFormat === 'left_space') return symbol + ' ' + formatted;
        if (posFormat === 'right') return formatted + symbol;
        if (posFormat === 'right_space') return formatted + ' ' + symbol;
        return symbol + formatted;
    }

    // Decode HTML entities (e.g. &#36; -> $)
    function decodeHtml(html) {
        var el = document.createElement('textarea');
        el.innerHTML = html;
        return el.value;
    }

    // Split bot response into intro and closing (stripping numbered product lists)
    function splitBotResponse(content) {
        var lines = content.split('\n');
        var intro = [];
        var closing = [];
        var foundNumbered = false;
        var lastNumberedIdx = -1;
        var firstNumberedIdx = -1;

        // Find numbered list lines (e.g. "1. Product..." or "2) Product...")
        for (var i = 0; i < lines.length; i++) {
            if (/^\s*\d+[\.\)]\s/.test(lines[i])) {
                if (firstNumberedIdx === -1) firstNumberedIdx = i;
                lastNumberedIdx = i;
                foundNumbered = true;
            }
        }

        if (!foundNumbered) {
            // No numbered list — split by double newline
            var parts = content.split(/\n\n+/);
            if (parts.length <= 1) {
                return { intro: content.trim(), closing: '' };
            }
            return { intro: parts[0].trim(), closing: parts.slice(1).join('\n\n').trim() };
        }

        // Collect lines before first numbered item (intro)
        intro = lines.slice(0, firstNumberedIdx).join('\n').trim();
        // Collect lines after last numbered item (closing)
        closing = lines.slice(lastNumberedIdx + 1).join('\n').trim();

        return { intro: intro, closing: closing };
    }

    // Add message to chat
    function addMessage(content, type, products, skipSave) {
        // If bot has products, render: intro bubble → product cards → closing bubble
        if (type === 'bot' && products && products.length > 0) {
            var parts = splitBotResponse(content);

            // Intro bubble (before cards)
            if (parts.intro && parts.intro.trim()) {
                var textDiv = document.createElement('div');
                textDiv.className = 'wccb-message wccb-bot';
                var mc = document.createElement('div');
                mc.className = 'wccb-message-content';
                mc.innerHTML = formatBotMessage(parts.intro);
                textDiv.appendChild(mc);
                messages.appendChild(textDiv);
            }

            // Product cards
            products.forEach(function(product) {
                if (!product.name && !product.price && !product.image) return;
                var cardWrap = document.createElement('div');
                cardWrap.className = 'wccb-product-card-wrap';

                var card = document.createElement('div');
                card.className = 'wccb-product-card';

                // Top row: image left, price+button right
                var topRow = document.createElement('div');
                topRow.className = 'wccb-product-card-top';

                if (product.image) {
                    var img = document.createElement('img');
                    img.src = product.image;
                    img.alt = product.name;
                    img.className = 'wccb-product-card-image';
                    topRow.appendChild(img);
                }

                var rightCol = document.createElement('div');
                rightCol.className = 'wccb-product-card-right';

                if (product.price) {
                    var priceEl = document.createElement('div');
                    priceEl.className = 'wccb-product-card-price';
                    var priceText = decodeHtml(formatPrice(product.price));
                    var priceSuffix = (wccb_widget.price_suffix || '').replace(/\{[^}]+\}/g, '').trim();
                    if (priceSuffix) priceText += ' ' + priceSuffix;
                    priceEl.textContent = priceText;
                    rightCol.appendChild(priceEl);
                }

                if (product.url) {
                    var link = document.createElement('a');
                    link.href = product.url;
                    link.className = 'wccb-product-card-btn';
                    link.target = '_self';
                    link.textContent = wccb_widget.strings.view_product || 'View Product';
                    rightCol.appendChild(link);
                }

                topRow.appendChild(rightCol);
                card.appendChild(topRow);

                // Bottom: title full width
                var nameEl = document.createElement('div');
                nameEl.className = 'wccb-product-card-name';
                nameEl.textContent = product.name;
                card.appendChild(nameEl);

                cardWrap.appendChild(card);
                messages.appendChild(cardWrap);
            });

            // Closing bubble (after cards)
            if (parts.closing && parts.closing.trim()) {
                var closingDiv = document.createElement('div');
                closingDiv.className = 'wccb-message wccb-bot';
                var closingMc = document.createElement('div');
                closingMc.className = 'wccb-message-content';
                closingMc.innerHTML = formatBotMessage(parts.closing);
                closingDiv.appendChild(closingMc);
                messages.appendChild(closingDiv);
            }

            scrollToBottom();
            if (!skipSave) {
                saveMessageToStorage(content, type, products);
            }
            return;
        }

        // Skip empty bot messages
        if (type === 'bot' && (!content || !content.trim())) return;

        var div = document.createElement('div');
        div.className = 'wccb-message wccb-' + type;

        var messageContent = document.createElement('div');
        messageContent.className = 'wccb-message-content';

        if (type === 'bot') {
            messageContent.innerHTML = formatBotMessage(content);
        } else {
            messageContent.textContent = content;
        }

        div.appendChild(messageContent);

        messages.appendChild(div);
        scrollToBottom();

        // Save to sessionStorage (skip when restoring)
        if (!skipSave) {
            saveMessageToStorage(content, type, products);
        }
    }

    // ---- Session persistence ----

    function saveChatOpen(open) {
        try { sessionStorage.setItem(STORAGE_OPEN, open ? '1' : '0'); } catch(e) {}
    }

    function saveMessageToStorage(content, type, products) {
        try {
            var stored = JSON.parse(sessionStorage.getItem(STORAGE_MESSAGES) || '[]');
            stored.push({ content: content, type: type, products: products || null });
            // Keep last 50 messages max
            if (stored.length > 50) stored = stored.slice(-50);
            sessionStorage.setItem(STORAGE_MESSAGES, JSON.stringify(stored));
        } catch(e) {}
    }

    function restoreChatState() {
        try {
            // Restore open state
            var wasOpen = sessionStorage.getItem(STORAGE_OPEN);
            if (wasOpen === '1') {
                isOpen = true;
                widget.classList.add('open');
            }

            // Restore messages
            var stored = JSON.parse(sessionStorage.getItem(STORAGE_MESSAGES) || '[]');
            if (stored.length > 0) {
                // Remove the default welcome message since we'll restore from storage
                var welcomeMsg = messages.querySelector('.wccb-message');
                if (welcomeMsg) {
                    welcomeMsg.remove();
                }

                for (var i = 0; i < stored.length; i++) {
                    var msg = stored[i];
                    addMessage(msg.content, msg.type, msg.products, true);
                }

                // Visitor returned with previous messages — start heartbeat
                startHeartbeat();
            }
        } catch(e) {}
    }

    // ---- End persistence ----

    // Show typing indicator
    function showTyping() {
        var div = document.createElement('div');
        div.id = 'wccb-typing-indicator';
        div.className = 'wccb-message wccb-bot';
        div.innerHTML = '<div class="wccb-typing"><span></span><span></span><span></span></div>';
        messages.appendChild(div);
        scrollToBottom();
    }

    // Hide typing indicator
    function hideTyping() {
        var typing = document.getElementById('wccb-typing-indicator');
        if (typing) {
            typing.remove();
        }
    }

    // Scroll to bottom
    function scrollToBottom() {
        messages.scrollTop = messages.scrollHeight;
    }

    // Cookie helpers
    function setCookie(name, value, days) {
        var expires = '';
        if (days) {
            var date = new Date();
            date.setTime(date.getTime() + (days * 24 * 60 * 60 * 1000));
            expires = '; expires=' + date.toUTCString();
        }
        document.cookie = name + '=' + (value || '') + expires + '; path=/';
    }

    function getCookie(name) {
        var nameEQ = name + '=';
        var ca = document.cookie.split(';');
        for (var i = 0; i < ca.length; i++) {
            var c = ca[i];
            while (c.charAt(0) === ' ') c = c.substring(1, c.length);
            if (c.indexOf(nameEQ) === 0) return c.substring(nameEQ.length, c.length);
        }
        return null;
    }

    // Darken color helper
    function darkenColor(color, percent) {
        var num = parseInt(color.replace('#', ''), 16);
        var amt = Math.round(2.55 * percent);
        var R = (num >> 16) - amt;
        var G = (num >> 8 & 0x00FF) - amt;
        var B = (num & 0x0000FF) - amt;

        R = R < 0 ? 0 : R;
        G = G < 0 ? 0 : G;
        B = B < 0 ? 0 : B;

        return '#' + (0x1000000 + R * 0x10000 + G * 0x100 + B).toString(16).slice(1);
    }

    // Handle escape key
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape' && isOpen) {
            isOpen = false;
            widget.classList.remove('open');
            saveChatOpen(false);
        }
    });

    // ---- Human Mode Polling ----
    function startHumanPoll() {
        stopHumanPoll();
        humanPollTimer = setInterval(pollForHumanReply, 3000);
    }

    function stopHumanPoll() {
        if (humanPollTimer) {
            clearInterval(humanPollTimer);
            humanPollTimer = null;
        }
        humanPollInFlight = false;
    }

    function pollForHumanReply() {
        if (humanPollInFlight) return;
        humanPollInFlight = true;

        var formData = new FormData();
        formData.append('action', 'wccb_poll_chat');
        formData.append('nonce', wccb_widget.nonce);
        formData.append('session_id', sessionId);

        fetch(wccb_widget.ajax_url, {
            method: 'POST',
            body: formData
        })
        .then(function(r) { return r.json(); })
        .then(function(data) {
            humanPollInFlight = false;
            if (!data.success) return;

            var d = data.data;

            // If human mode was turned off, stop polling
            if (!d.human_mode) {
                stopHumanPoll();
                waitingForHumanReply = false;
                return;
            }

            if (!waitingForHumanReply) return;

            // Check if the last message in history is an assistant reply
            var history = d.history || [];
            if (history.length === 0) return;

            var lastMsg = history[history.length - 1];
            if (lastMsg.role === 'assistant') {
                waitingForHumanReply = false;
                stopHumanPoll();
                addMessage(lastMsg.content, 'bot');
            }
        })
        .catch(function() {
            humanPollInFlight = false;
        });
    }

    // ---- Heartbeat (visitor presence) ----
    function sendHeartbeat() {
        var fd = new FormData();
        fd.append('action', 'wccb_heartbeat');
        fd.append('session_id', sessionId);
        fetch(wccb_widget.ajax_url, { method: 'POST', body: fd, keepalive: true })
            .catch(function() {});
    }

    function startHeartbeat() {
        if (heartbeatTimer) return;
        heartbeatStarted = true;
        sendHeartbeat();
        heartbeatTimer = setInterval(sendHeartbeat, 15000);
    }

    function stopHeartbeat() {
        if (heartbeatTimer) {
            clearInterval(heartbeatTimer);
            heartbeatTimer = null;
        }
    }

    // ---- Visitor Tracking ----
    var trackSent = false;
    var chatTrackSent = false;
    var visitToken = 'vt-' + Math.random().toString(36).substr(2, 9) + Date.now().toString(36);
    var pageLoadTime = Date.now();

    // Track page visit on load
    function trackVisit(isChat) {
        var formData = new FormData();
        formData.append('action', 'wccb_track');
        formData.append('page_url', window.location.href);
        formData.append('is_chat', isChat ? '1' : '0');
        formData.append('visit_token', visitToken);
        fetch(wccb_widget.ajax_url, {
            method: 'POST',
            body: formData,
            keepalive: true
        }).catch(function() {});
    }

    // Send duration on page unload
    function sendDuration() {
        var duration = Math.round((Date.now() - pageLoadTime) / 1000);
        if (duration < 1) return;
        var formData = new FormData();
        formData.append('action', 'wccb_track_duration');
        formData.append('visit_token', visitToken);
        formData.append('duration', duration.toString());
        if (navigator.sendBeacon) {
            navigator.sendBeacon(wccb_widget.ajax_url, formData);
        } else {
            fetch(wccb_widget.ajax_url, {
                method: 'POST',
                body: formData,
                keepalive: true
            }).catch(function() {});
        }
    }

    // Fire pageview beacon once
    if (!trackSent) {
        trackSent = true;
        trackVisit(false);
    }

    // Send duration when leaving + pause/resume heartbeat
    document.addEventListener('visibilitychange', function() {
        if (document.visibilityState === 'hidden') {
            sendDuration();
            stopHeartbeat();
        } else if (document.visibilityState === 'visible' && heartbeatStarted) {
            startHeartbeat();
        }
    });
    window.addEventListener('pagehide', function() {
        sendDuration();
        stopHeartbeat();
    });

    // Intercept first chat message to also fire a chat tracking beacon + start heartbeat
    var originalSendMessage = sendMessage;
    sendMessage = function(message) {
        if (!chatTrackSent) {
            chatTrackSent = true;
            trackVisit(true);
        }
        if (!heartbeatStarted) {
            startHeartbeat();
        }
        originalSendMessage(message);
    };

})();
