/**
 * WC Chatbot Admin JavaScript
 */

(function($) {
    'use strict';

    $(document).ready(function() {
        // Tab switching
        $('.wccb-tabs .nav-tab').on('click', function(e) {
            e.preventDefault();
            var tab = $(this).data('tab');
            $('.wccb-tabs .nav-tab').removeClass('nav-tab-active');
            $(this).addClass('nav-tab-active');
            $('.wccb-tab-panel').hide();
            $('#wccb-tab-' + tab).show();
            // Re-init Select2 when switching to bot-training tab (fixes hidden container width)
            if (tab === 'bot-training' && $.fn.select2 && $('.wccb-select2').length) {
                $('.wccb-select2').select2({
                    placeholder: wccb_admin.strings.search_language || 'Search for a language...',
                    allowClear: true,
                    width: '100%'
                });
            }
        });

        // Initialize Select2 for language selector
        if ($.fn.select2 && $('.wccb-select2').length) {
            $('.wccb-select2').select2({
                placeholder: wccb_admin.strings.search_language || 'Search for a language...',
                allowClear: true,
                width: '100%'
            });
        }

        // Silent ping: send plugin version to hub on page load
        if ($('#license_key').val()) {
            $.post(wccb_admin.ajax_url, {
                action: 'wccb_ping_version',
                nonce: wccb_admin.nonce
            });
        }

        // Test Connection
        $('#wccb-test-connection').on('click', function() {
            var $btn = $(this);
            var $result = $('#wccb-test-result');
            var $info = $('#wccb-license-info');

            var licenseKey = $('#license_key').val();

            if (!licenseKey) {
                $result.removeClass('loading success').addClass('error').text(wccb_admin.strings.enter_license_first);
                $info.hide();
                return;
            }

            $btn.prop('disabled', true);
            $result.removeClass('success error').addClass('loading').text(wccb_admin.strings.testing);
            $info.hide();

            $.ajax({
                url: wccb_admin.ajax_url,
                type: 'POST',
                data: {
                    action: 'wccb_test_connection',
                    nonce: wccb_admin.nonce,
                    license_key: licenseKey
                },
                success: function(response) {
                    if (response.success) {
                        var d = response.data;
                        $result.removeClass('loading').addClass('success').text(d.message);

                        // Show license info
                        var statusLabels = {
                            'trial': 'Trial',
                            'active': 'Active',
                            'expired': 'Expired',
                            'suspended': 'Suspended'
                        };
                        var statusColors = {
                            'trial': '#dba617',
                            'active': '#00a32a',
                            'expired': '#d63638',
                            'suspended': '#d63638'
                        };

                        var status = d.status || 'unknown';
                        var statusLabel = statusLabels[status] || status;
                        var statusColor = statusColors[status] || '#666';

                        $('#wccb-info-status').html(
                            '<span style="display:inline-block;padding:2px 10px;border-radius:12px;font-size:13px;font-weight:600;color:#fff;background:' + statusColor + ';">' + statusLabel + '</span>'
                        );

                        var expiresAt = d.expires_at;
                        if (expiresAt) {
                            var expDate = new Date(expiresAt);
                            var formatted = expDate.toLocaleDateString(undefined, { day: '2-digit', month: '2-digit', year: 'numeric' });
                            $('#wccb-info-expires').text(formatted);
                        } else {
                            $('#wccb-info-expires').text(wccb_admin.strings.no_expiration);
                        }

                        // Update sidebar with fresh data from Hub
                        $('#wccb-sidebar-chats').text((d.chats_used || 0) + ' / ' + (d.monthly_limit || 50));

                        $info.show();
                    } else {
                        $result.removeClass('loading').addClass('error').text(wccb_admin.strings.error + ' ' + response.data.message);
                        $info.hide();
                    }
                },
                error: function() {
                    $result.removeClass('loading').addClass('error').text(wccb_admin.strings.error + ' ' + wccb_admin.strings.connection_failed);
                    $info.hide();
                },
                complete: function() {
                    $btn.prop('disabled', false);
                }
            });
        });

        // Sync Now (batched with progress)
        $('#wccb-sync-now').on('click', function() {
            var $btn = $(this);
            var $result = $('#wccb-sync-result');
            var batchSize = 100;

            $btn.prop('disabled', true);
            $result.removeClass('success error').addClass('loading').text(wccb_admin.strings.preparing_sync);

            // Step 1: Get total product count
            $.ajax({
                url: wccb_admin.ajax_url,
                type: 'POST',
                data: {
                    action: 'wccb_sync_batch',
                    nonce: wccb_admin.nonce,
                    step: 'count'
                },
                success: function(response) {
                    if (response.success && response.data.total > 0) {
                        syncBatch(0, batchSize, response.data.total, $btn, $result);
                    } else if (response.success && response.data.total === 0) {
                        $result.removeClass('loading').addClass('error').text(wccb_admin.strings.no_products);
                        $btn.prop('disabled', false);
                    } else {
                        $result.removeClass('loading').addClass('error').text(wccb_admin.strings.error + ' ' + (response.data ? response.data.message : 'Failed'));
                        $btn.prop('disabled', false);
                    }
                },
                error: function() {
                    $result.removeClass('loading').addClass('error').text(wccb_admin.strings.error + ' ' + wccb_admin.strings.connection_error);
                    $btn.prop('disabled', false);
                }
            });
        });

        // Vertical sub-tab switching (Bot Training)
        $('.wccb-subtab-link').on('click', function(e) {
            e.preventDefault();
            var subtab = $(this).data('subtab');
            $('.wccb-subtab-link').removeClass('active');
            $(this).addClass('active');
            $('.wccb-subtab-panel').hide();
            $('#wccb-subtab-' + subtab).show();
            // Re-init Select2 when switching to main-training sub-tab
            if (subtab === 'main-training' && $.fn.select2 && $('.wccb-select2').length) {
                $('.wccb-select2').select2({
                    placeholder: wccb_admin.strings.search_language || 'Search for a language...',
                    allowClear: true,
                    width: '100%'
                });
            }
        });

        // Generate Prompt (inline in sub-tab)
        $('#wccb-generate-prompt').on('click', function() {
            var $btn = $(this);
            var $result = $('#wccb-generate-result');

            var answers = {};
            for (var i = 1; i <= 14; i++) {
                if (i === 13) {
                    // Q13 is language checkboxes + other field
                    var langs = [];
                    $('.wccb-q13-cb:checked').each(function() { langs.push($(this).val()); });
                    var otherLangs = $('#wccb-q13-other').val();
                    if (otherLangs) langs.push(otherLangs.trim());
                    answers['q13'] = langs.join(', ');
                } else {
                    var $field = $('#wccb-q' + i);
                    answers['q' + i] = $field.val() || '';
                }
            }

            $btn.prop('disabled', true).text(wccb_admin.strings.generating);
            $result.removeClass('success error').addClass('loading').text(wccb_admin.strings.ai_generating).show();

            $.ajax({
                url: wccb_admin.ajax_url,
                type: 'POST',
                data: $.extend({
                    action: 'wccb_generate_prompt',
                    nonce: wccb_admin.nonce
                }, answers),
                success: function(response) {
                    if (response.success && response.data.prompt) {
                        $('#custom_prompt').val(response.data.prompt);
                        $result.removeClass('loading').addClass('success').text(wccb_admin.strings.prompt_generated);
                        setTimeout(function() {
                            $result.hide();
                            $('.wccb-subtab-link[data-subtab="main-training"]').trigger('click');
                        }, 1500);
                    } else {
                        var msg = response.data && response.data.message ? response.data.message : wccb_admin.strings.prompt_failed;
                        $result.removeClass('loading').addClass('error').text(wccb_admin.strings.error + ' ' + msg);
                    }
                },
                error: function() {
                    $result.removeClass('loading').addClass('error').text(wccb_admin.strings.error + ' ' + wccb_admin.strings.connection_failed);
                },
                complete: function() {
                    $btn.prop('disabled', false).text(wccb_admin.strings.generate_prompt_label);
                }
            });
        });

        // Add More Data: Add new training entry
        $('#wccb-add-training-entry').on('click', function() {
            var $container = $('#wccb-training-entries');
            var index = $container.find('.wccb-training-entry').length;
            var removeLabel = (wccb_admin.strings && wccb_admin.strings.remove) || 'Remove';
            var topicLabel = (wccb_admin.strings && wccb_admin.strings.topic) || 'Topic';
            var contentLabel = (wccb_admin.strings && wccb_admin.strings.content) || 'Content';

            var html = '<div class="wccb-training-entry" data-index="' + index + '">'
                + '<div class="wccb-training-entry-header">'
                + '<span class="wccb-training-entry-number">#' + (index + 1) + '</span>'
                + '<button type="button" class="button button-link-delete wccb-remove-entry">' + removeLabel + '</button>'
                + '</div>'
                + '<div class="wccb-training-entry-fields">'
                + '<label>' + topicLabel + '</label>'
                + '<input type="text" name="wccb_settings[additional_training][' + index + '][topic]" class="regular-text" placeholder="' + (wccb_admin.strings.placeholder_topic || 'e.g. Return Policy, Warranty Info, Store Hours...') + '">'
                + '<label>' + contentLabel + '</label>'
                + '<textarea name="wccb_settings[additional_training][' + index + '][content]" rows="4" class="large-text" placeholder="' + (wccb_admin.strings.placeholder_content || 'Enter the training information here...') + '"></textarea>'
                + '</div>'
                + '</div>';

            $container.append(html);
        });

        // Add More Data: Remove training entry and re-index
        $(document).on('click', '.wccb-remove-entry', function() {
            $(this).closest('.wccb-training-entry').remove();
            $('#wccb-training-entries .wccb-training-entry').each(function(i) {
                $(this).attr('data-index', i);
                $(this).find('.wccb-training-entry-number').text('#' + (i + 1));
                $(this).find('input, textarea').each(function() {
                    var name = $(this).attr('name');
                    if (name) {
                        name = name.replace(/\[additional_training\]\[\d+\]/, '[additional_training][' + i + ']');
                        $(this).attr('name', name);
                    }
                });
            });
        });

        // Icon style visual selector
        $('.wccb-icon-option').on('click', function() {
            var $this = $(this);
            var value = $this.data('value');
            $('.wccb-icon-option').removeClass('active');
            $this.addClass('active');
            $('#widget_icon_style').val(value);
        });

        // Color picker live hex update
        $('.wccb-color-field input[type="color"]').on('input', function() {
            $(this).siblings('.wccb-color-hex').text($(this).val());
        });

        // ---- Chat History ----
        var chatHistoryLoaded = false;
        var chatHistoryData = [];

        function loadChatHistory(page, silent) {
            page = page || 1;
            if (!silent) {
                $('#wccb-chat-history-loading').show();
                $('#wccb-chat-history-content, #wccb-chat-history-empty, #wccb-chat-history-error').hide();
            }

            $.ajax({
                url: wccb_admin.ajax_url,
                type: 'POST',
                data: { action: 'wccb_fetch_chat_history', nonce: wccb_admin.nonce, page: page, per_page: 20 },
                success: function(response) {
                    if (!silent) $('#wccb-chat-history-loading').hide();
                    if (response.success) {
                        chatHistoryData = response.data.logs || [];
                        var pagination = response.data.pagination || {};

                        if (chatHistoryData.length === 0) {
                            $('#wccb-chat-history-empty').show();
                            return;
                        }

                        var tbody = '';
                        chatHistoryData.forEach(function(log, idx) {
                            var date = formatDateTz(log.created_at || '');
                            var ip = log.visitor_ip || '-';
                            var session = (log.session_id || '').substring(0, 12) + '...';
                            var msgs = log.message_count || 0;
                            tbody += '<tr>';
                            tbody += '<td>' + escHtml(date) + '</td>';
                            tbody += '<td>' + escHtml(ip) + '</td>';
                            tbody += '<td><code>' + escHtml(session) + '</code></td>';
                            tbody += '<td>' + msgs + '</td>';
                            tbody += '<td><button type="button" class="button button-small wccb-view-chat" data-idx="' + idx + '">' + wccb_admin.strings.view + '</button> <button type="button" class="button button-small wccb-delete-chat" data-id="' + (log.id || '') + '" data-idx="' + idx + '" title="' + (wccb_admin.strings.delete_chat || 'Delete') + '" style="color:#d63638;border-color:#d63638;min-width:0;padding:0 6px;"><span class="dashicons dashicons-trash" style="font-size:16px;line-height:26px;width:16px;height:16px;"></span></button></td>';
                            tbody += '</tr>';
                        });

                        $('#wccb-chat-history-body').html(tbody);

                        // Pagination
                        var pagesHtml = '';
                        if (pagination.total_pages > 1) {
                            pagesHtml += '<span class="displaying-num">' + pagination.total + ' ' + wccb_admin.strings.items + '</span> ';
                            for (var p = 1; p <= pagination.total_pages; p++) {
                                if (p === pagination.page) {
                                    pagesHtml += '<span class="tablenav-pages-navspan button disabled">' + p + '</span> ';
                                } else {
                                    pagesHtml += '<a class="button wccb-chat-page" data-page="' + p + '" href="#">' + p + '</a> ';
                                }
                            }
                        }
                        $('#wccb-chat-pages').html(pagesHtml);

                        $('#wccb-chat-history-content').show();
                        chatHistoryLoaded = true;
                    } else if (!silent) {
                        $('#wccb-chat-history-error').text(response.data.message || wccb_admin.strings.history_error).show();
                    }
                },
                error: function() {
                    if (!silent) {
                        $('#wccb-chat-history-loading').hide();
                        $('#wccb-chat-history-error').text(wccb_admin.strings.connection_error).show();
                    }
                }
            });
        }

        // Load on tab switch (always reload to get fresh data)
        $(document).on('click', '.nav-tab[data-tab="chat-history"]', function() {
            loadChatHistory(1);
        });

        // Pagination click
        $(document).on('click', '.wccb-chat-page', function(e) {
            e.preventDefault();
            loadChatHistory($(this).data('page'));
        });

        // Split bot text into intro/closing (strip numbered product lists)
        function splitBotText(content) {
            var lines = content.split('\n');
            var firstNumbered = -1, lastNumbered = -1;
            for (var i = 0; i < lines.length; i++) {
                if (/^\s*\d+[\.\)]\s/.test(lines[i])) {
                    if (firstNumbered === -1) firstNumbered = i;
                    lastNumbered = i;
                }
            }
            if (firstNumbered === -1) {
                var parts = content.split(/\n\n+/);
                if (parts.length <= 1) return { intro: content, closing: '' };
                return { intro: parts[0], closing: parts.slice(1).join('\n\n') };
            }
            var intro = lines.slice(0, firstNumbered).join('\n').trim();
            var closing = lines.slice(lastNumbered + 1).join('\n').trim();
            return { intro: intro, closing: closing };
        }

        // Render chat messages HTML (reusable)
        function renderChatHtml(history) {
            var html = '';
            var pSuffix = (wccb_admin.price_suffix || '').replace(/\{[^}]+\}/g, '').trim();

            (history || []).forEach(function(msg) {
                if (msg.role === 'user') {
                    html += '<div class="wccb-chat-msg-user"><div class="wccb-chat-msg-bubble-user">' + formatMd(escHtml(msg.content || '')) + '</div></div>';
                } else {
                    var hasProducts = msg.products && msg.products.length > 0;
                    var rawContent = msg.content || '';

                    if (hasProducts) {
                        var parts = splitBotText(rawContent);
                        if (parts.intro) {
                            html += '<div class="wccb-chat-msg-bot"><div class="wccb-chat-msg-bubble-bot">' + formatMd(escHtml(parts.intro)) + '</div></div>';
                        }
                        msg.products.forEach(function(product) {
                            var pPrice = String(product.price || '');
                            html += '<table cellpadding="0" cellspacing="0" border="0" style="border:1px solid #e5e7eb;border-radius:12px;overflow:hidden;background:#fff;max-width:75%;margin:4px 0;border-collapse:separate;table-layout:auto;"><tr>';
                            if (product.image) html += '<td style="width:80px;vertical-align:top;padding:0;"><img src="' + escHtml(product.image) + '" width="80" height="80" style="display:block;width:80px;height:80px;object-fit:cover;" alt=""></td>';
                            html += '<td style="vertical-align:middle;padding:8px 12px;">';
                            if (pPrice) html += '<div style="font-size:15px;font-weight:700;color:#4F46E5;margin-bottom:4px;">' + escHtml(formatPrice(pPrice)) + (pSuffix ? ' ' + escHtml(pSuffix) : '') + '</div>';
                            if (product.url) html += '<a href="' + escHtml(product.url) + '" target="_blank" style="display:inline-block;padding:4px 12px;background:#4F46E5;color:#fff;text-decoration:none;border-radius:6px;font-size:11px;font-weight:600;">' + (wccb_admin.strings.view_product || 'Ver Producto') + '</a>';
                            html += '</td></tr>';
                            if (product.name) html += '<tr><td colspan="2" style="padding:6px 10px;border-top:1px solid #e5e7eb;background:#f9fafb;font-size:11px;font-weight:600;color:#1d2327;">' + escHtml(product.name) + '</td></tr>';
                            html += '</table>';
                        });
                        if (parts.closing) {
                            html += '<div class="wccb-chat-msg-bot"><div class="wccb-chat-msg-bubble-bot">' + formatMd(escHtml(parts.closing)) + '</div></div>';
                        }
                    } else {
                        html += '<div class="wccb-chat-msg-bot"><div class="wccb-chat-msg-bubble-bot">' + formatMd(escHtml(rawContent)) + '</div></div>';
                    }
                }
            });

            // Typing indicator: last message is from user = bot is thinking
            if (history && history.length > 0 && history[history.length - 1].role === 'user') {
                html += '<div class="wccb-chat-msg-bot"><div style="display:inline-flex;align-items:center;gap:4px;padding:10px 14px;background:#f0f0f0;border-radius:12px;"><span class="wccb-typing-dot"></span><span class="wccb-typing-dot"></span><span class="wccb-typing-dot"></span></div></div>';
            }

            return html;
        }

        // Live modal polling state
        var liveModalChatId = null;
        var liveModalTimer = null;
        var liveModalMsgCount = 0;

        // View conversation
        $(document).on('click', '.wccb-view-chat', function() {
            var idx = $(this).data('idx');
            var log = chatHistoryData[idx];
            if (!log) return;

            var meta = '<strong>Date:</strong> ' + escHtml(formatDateTz(log.created_at || '')) +
                ' &nbsp; <strong>IP:</strong> ' + escHtml(log.visitor_ip || '-') +
                ' &nbsp; <strong>Session:</strong> <code>' + escHtml(log.session_id || '') + '</code>';
            $('#wccb-chat-detail-meta').html(meta);
            $('#wccb-chat-detail-messages').html(renderChatHtml(log.history || []));
            $('#wccb-chat-detail-modal').fadeIn(200);

            // Start live polling
            liveModalChatId = log.id;
            liveModalMsgCount = (log.history || []).length;
            startModalPolling();
        });

        function startModalPolling() {
            stopModalPolling();
            liveModalTimer = setInterval(pollModalChat, 3000);
        }
        function stopModalPolling() {
            if (liveModalTimer) { clearInterval(liveModalTimer); liveModalTimer = null; }
            liveModalChatId = null;
        }
        function pollModalChat() {
            if (!liveModalChatId) return;
            $.ajax({
                url: wccb_admin.ajax_url,
                type: 'POST',
                data: { action: 'wccb_fetch_single_chat', nonce: wccb_admin.nonce, chat_log_id: liveModalChatId },
                success: function(response) {
                    if (response.success && response.data && response.data.log) {
                        var newHistory = response.data.log.history || [];
                        if (newHistory.length !== liveModalMsgCount) {
                            liveModalMsgCount = newHistory.length;
                            var container = $('#wccb-chat-detail-messages');
                            container.html(renderChatHtml(newHistory));
                            container[0].scrollTop = container[0].scrollHeight;
                        }
                    }
                }
            });
        }

        // Close chat detail modal
        $('#wccb-chat-detail-modal .wccb-modal-close').on('click', function() {
            $('#wccb-chat-detail-modal').fadeOut(200);
            stopModalPolling();
        });
        $('#wccb-chat-detail-modal').on('click', function(e) {
            if ($(e.target).is('.wccb-modal-overlay')) {
                $(this).fadeOut(200);
                stopModalPolling();
            }
        });

        // Delete chat conversation
        $(document).on('click', '.wccb-delete-chat', function() {
            var chatId = $(this).data('id');
            if (!chatId) return;
            if (!confirm(wccb_admin.strings.confirm_delete || 'Delete this conversation?')) return;

            var $btn = $(this);
            $btn.prop('disabled', true);

            $.ajax({
                url: wccb_admin.ajax_url,
                type: 'POST',
                data: { action: 'wccb_delete_chat', nonce: wccb_admin.nonce, chat_log_id: chatId },
                success: function(response) {
                    if (response.success) {
                        loadChatHistory(1);
                    } else {
                        alert(response.data.message || 'Error');
                        $btn.prop('disabled', false);
                    }
                },
                error: function() {
                    alert('Connection error');
                    $btn.prop('disabled', false);
                }
            });
        });

        // Auto-refresh chat history every 10 seconds (silent — no loading flash)
        var chatRefreshTimer = null;

        function startChatRefresh() {
            if (chatRefreshTimer) return;
            chatRefreshTimer = setInterval(function() {
                if ($('#wccb-chat-detail-modal').is(':visible')) return;
                loadChatHistory(1, true);
            }, 10000);
        }

        function stopChatRefresh() {
            if (chatRefreshTimer) { clearInterval(chatRefreshTimer); chatRefreshTimer = null; }
        }

        // Start refresh when switching to chat-history tab, stop on other tabs
        $(document).on('click', '.nav-tab[data-tab="chat-history"]', function() {
            startChatRefresh();
        });
        $(document).on('click', '.nav-tab:not([data-tab="chat-history"])', function() {
            stopChatRefresh();
        });

        // Also start immediately if chat-history tab is already active on page load
        if ($('.nav-tab[data-tab="chat-history"]').hasClass('nav-tab-active')) {
            startChatRefresh();
        }

        function escHtml(str) {
            var div = document.createElement('div');
            div.appendChild(document.createTextNode(str));
            return div.innerHTML;
        }

        // Convert markdown bold + links to HTML (call AFTER escHtml)
        function formatMd(text) {
            return text
                .replace(/\*\*(.+?)\*\*/g, '<strong>$1</strong>')
                .replace(/\[([^\]]+)\]\(([^)]+)\)/g, '<a href="$2" target="_blank" style="color:#4F46E5;">$1</a>')
                .replace(/\n/g, '<br>');
        }

        // Format price using WooCommerce settings (same logic as widget.js)
        function decodeHtmlEntities(str) {
            var el = document.createElement('textarea');
            el.innerHTML = str;
            return el.value;
        }
        function formatPrice(price) {
            var pf = (wccb_admin && wccb_admin.price_format) || {};
            var symbol = decodeHtmlEntities(pf.currency_symbol || '$');
            var thousandSep = pf.thousand_sep || ',';
            var decimalSep = pf.decimal_sep || '.';
            var decimals = typeof pf.num_decimals !== 'undefined' ? parseInt(pf.num_decimals) : 2;
            var posFormat = pf.currency_pos || 'left';

            var num = parseFloat(price);
            if (isNaN(num)) return price;

            var fixed = num.toFixed(decimals);
            var parts = fixed.split('.');
            var intPart = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, thousandSep);
            var formatted = decimals > 0 ? intPart + decimalSep + parts[1] : intPart;

            if (posFormat === 'left') return symbol + formatted;
            if (posFormat === 'left_space') return symbol + ' ' + formatted;
            if (posFormat === 'right') return formatted + symbol;
            if (posFormat === 'right_space') return formatted + ' ' + symbol;
            return symbol + formatted;
        }

        // Format a UTC date string to the configured timezone
        function formatDateTz(dateStr) {
            if (!dateStr) return '';
            var tz = (wccb_admin && wccb_admin.timezone) ? wccb_admin.timezone : 'UTC';
            try {
                var d = new Date(dateStr.replace(' ', 'T') + (dateStr.indexOf('Z') === -1 && dateStr.indexOf('+') === -1 ? 'Z' : ''));
                return d.toLocaleString('default', {
                    timeZone: tz,
                    year: 'numeric', month: '2-digit', day: '2-digit',
                    hour: '2-digit', minute: '2-digit', hour12: false
                });
            } catch (e) {
                return dateStr;
            }
        }

        // ---- Sync ----
        var syncCounter = 0;
        var syncTimer = null;

        function animateCounter(target, total, $result) {
            if (syncTimer) clearInterval(syncTimer);
            var speed = Math.max(5, Math.floor(200 / (target - syncCounter)));
            syncTimer = setInterval(function() {
                syncCounter++;
                $result.text(wccb_admin.strings.syncing_progress.replace('%1$s', syncCounter).replace('%2$s', total));
                if (syncCounter >= target) {
                    clearInterval(syncTimer);
                    syncTimer = null;
                }
            }, speed);
        }

        function syncBatch(offset, batchSize, total, $btn, $result) {
            var target = Math.min(offset + batchSize, total);
            animateCounter(target, total, $result);

            $.ajax({
                url: wccb_admin.ajax_url,
                type: 'POST',
                data: {
                    action: 'wccb_sync_batch',
                    nonce: wccb_admin.nonce,
                    step: 'batch',
                    offset: offset,
                    limit: batchSize,
                    is_first: offset === 0 ? 1 : 0,
                    is_last: (offset + batchSize >= total) ? 1 : 0,
                    total: total
                },
                success: function(response) {
                    if (response.success) {
                        if (offset + batchSize < total) {
                            syncBatch(offset + batchSize, batchSize, total, $btn, $result);
                        } else {
                            if (syncTimer) clearInterval(syncTimer);
                            syncCounter = 0;
                            $result.removeClass('loading').addClass('success').text(wccb_admin.strings.synced_success.replace('%d', total));
                            setTimeout(function() { location.reload(); }, 1500);
                        }
                    } else {
                        if (syncTimer) clearInterval(syncTimer);
                        syncCounter = 0;
                        $result.removeClass('loading').addClass('error').text(wccb_admin.strings.error + ' ' + response.data.message);
                        $btn.prop('disabled', false);
                    }
                },
                error: function() {
                    if (syncTimer) clearInterval(syncTimer);
                    syncCounter = 0;
                    $result.removeClass('loading').addClass('error').text(wccb_admin.strings.error + ' ' + wccb_admin.strings.sync_failed_at + ' ' + target + '/' + total);
                    $btn.prop('disabled', false);
                }
            });
        }
    });

    // Quick Access - Manage Store button
    $('#wccb-manage-store').on('click', function(e) {
        e.preventDefault();
        var $btn = $(this);
        $btn.prop('disabled', true).css('opacity', '0.7');
        var origText = $btn.html();
        $btn.html('<span class="dashicons dashicons-update" style="animation:rotation 1s infinite linear;font-size:16px;width:16px;height:16px;margin-top:2px;"></span> ' + (wccb_admin.strings.loading || 'Loading...'));

        $.ajax({
            url: wccb_admin.ajax_url,
            type: 'POST',
            data: {
                action: 'wccb_quick_access',
                nonce: wccb_admin.nonce
            },
            success: function(response) {
                if (response.success && response.data && response.data.url) {
                    window.open(response.data.url, '_blank');
                } else {
                    alert(response.data && response.data.message ? response.data.message : 'Could not generate access link. Please try again.');
                }
                $btn.prop('disabled', false).css('opacity', '1').html(origText);
            },
            error: function() {
                alert('Connection error. Please try again.');
                $btn.prop('disabled', false).css('opacity', '1').html(origText);
            }
        });
    });

})(jQuery);
